# coding: utf-8

import logging
import requests
import re

log = logging.getLogger(__name__)


class JugglerClient():

    def __init__(self, host, service=None):
        self.host = host
        self.service = service

    def to_juggler_service_name(self, name):
        # replace symbols to "_" to satisfy https://a.yandex-team.ru/arc/trunk/arcadia/juggler/libjuggler/validators/system.py?blame=true&rev=3519248#L9
        return re.sub(r'[^\w\-+=\./\t ]', '_', name)

    def truncate_description_string(self, desc, maxlen=700):
        """
        truncate_description_string limit description length to maxlen

        :param desc: Original description
        :param maxlen: Max description length
        :return: returns a description limited to maxlen
        TODO: fair sting bytes length may be needed
        """
        shorten_str = ' ... '
        if len(desc) > maxlen:
            idx = int(maxlen/2) - int(len(shorten_str)/2)
            return desc[:idx] + shorten_str + desc[(len(desc) - idx + 1):]
        else:
            return desc

    def send_events_to_juggler(self, level, message, service_suffix=None):
        if service_suffix:
            full_service_name = self.to_juggler_service_name('_'.join([s for s in (self.service, service_suffix) if s is not None]))
        else:
            full_service_name = self.service
        log.debug('Juggler message:\nService %s\nLevel %s,\nMessage %s', full_service_name, level, message)
        try:
            reply = requests.post("http://juggler-push.search.yandex.net/events", json={
                "source": "sandbox",
                "events": [{
                    "host": self.host,
                    "service": full_service_name,
                    "status": level,
                    "description": self.truncate_description_string(message)
                }]
            })
            event_status = reply.json()["events"][0]
            if event_status["code"] != 200:
                raise Exception(event_status["error"])
        except Exception:
            logging.exception("unable to get frontend nodes from juggler")
