# -*- coding: utf-8 -*-

import logging
import json
import re
from collections import defaultdict
from datetime import datetime
from library.python import resource
from jinja2 import Environment, FunctionLoader
from dateutil import parser
from .constants import (
    PRIORITY_OF_OWNERS_FROM_ABC,
    HW_OWNERS_BY_ABC,
    SERVICE_OWNERS_BY_W_TAG,
    ABC_CNAME,
    ST_TAGS_BY_SCENARIO_TYPE,
    MULTI_DC_TICKET_PARENT_TAG,
    JANITOR_PROCESSING_TAG
    )
from infra.rtc.janitor.clients.bot_client import BotClientHostNotFound
from infra.rtc.janitor.exceptions import (
    InvalidParsedDataException
    )
from walle_api import WalleApiError


log = logging.getLogger(__name__)
fqdn_regex = re.compile(r'(?=^.{2,254}$)(^(?:(?!\d|-)[a-zA-Z0-9\-]{1,63}(?<!-)\.?)+\.[a-zA-Z]{2,3}$)')
inv_regex = re.compile(r'^[0-9]{5,30}$')


def render_template(tpl_name, **kwargs):
    def loader_template(tpl_name):
        return resource.find('{}/{}'.format('templates', tpl_name)).decode('utf8')

    env = Environment(loader=FunctionLoader(loader_template))
    log.debug("template: {}".format(tpl_name))
    template = env.get_template(tpl_name)
    return template.render(**kwargs).encode('utf8')


def add_st_comment(st_client, st_ticket, **kwargs):
    issue = st_client.issues[st_ticket]
    issue.comments.create(**kwargs)


def add_st_tags(st_client, st_ticket, tags):
    issue = st_client.issues[st_ticket]
    log.info('''Adding tags {} to ticket {}'''.format(repr(tags), st_ticket))
    issue.update(tags={'add': tags})


def remove_st_janitor_processing(st_client, st_ticket):
    log.info('''Removing Janitor processing marking tag from ticket {}'''.format(st_ticket))
    remove_st_tags(st_client, st_ticket, (JANITOR_PROCESSING_TAG, ))


def remove_st_tags(st_client, st_ticket, tags):
    issue = st_client.issues[st_ticket]
    log.info('''Removing tags {} from ticket {}'''.format(repr(tags), st_ticket))
    issue.update(tags={'remove': tags})


def st_transistion(st_client, st_ticket, state, **kwargs):
    kwargs = {k: v for k, v in kwargs.items() if k in ('comment', 'resolution')}
    issue = st_client.issues[st_ticket]
    if issue.status.key == state:
        add_st_comment(st_client, st_ticket, **kwargs)
        return  # already in this state
    avaliable_transitions = {s.to.key: s.id for s in issue.transitions.get_all()}
    if state not in avaliable_transitions:
        raise Exception('Can not transit issue to state {}. Avaliable transitions are (state: transition) {}'.format(state, repr(avaliable_transitions)))
    transition = issue.transitions[avaliable_transitions[state]]
    transition.execute(**kwargs)


def st_set_assignee(st_client, st_ticket, assignee_login):
    issue = st_client.issues[st_ticket]
    issue.update(assignee=assignee_login)


def st_check_if_new_and_unassigned(st_client, st_ticket):
    issue = st_client.issues[st_ticket]
    if issue.assignee is None and issue.status.key == "new":
        return True
    else:
        return False


def st_create_link(st_client, st_ticket, anoter_st_ticket, relationship):
    issue = st_client.issues[st_ticket]
    issue.links.create(issue=anoter_st_ticket, relationship=relationship)


def st_get_linked_tickets(st_client, st_ticket):
    issue = st_client.issues[st_ticket]
    return [{'relationship': link.direction, 'ticket': link.object} for link in issue.links.get_all()]


def st_get_linked_multidc_subtickets(st_client, st_ticket, subticket_tag):
    res = []
    for subticket in st_get_linked_tickets(st_client, st_ticket):
        log.debug('ticket %s, linked ticket %s', st_ticket, subticket['ticket'].key)
        try:
            if subticket_tag in subticket['ticket'].tags:
                log.debug('%r -> %r, status %r', st_ticket, subticket['ticket'].key, subticket['ticket'].status.key)
                res.append(subticket)
            else:
                log.debug('%r -> %r skipped', st_ticket, subticket['ticket'].key)
        except Exception as e:
            log.error('%r -> %r access error: %s', st_ticket, subticket['ticket'].key, e)
            pass
    return res

# def filter_hosts()


def find_intersections_by_hosts(context, checked_scenario, status=None, janitor_owned=False, labels=None):
    """
        return: List
    """
    if status is None:
        status = ('created', 'started')
    if janitor_owned:
        match_labels = dict(checked_scenario.default_scenario_labels)
    else:
        match_labels = {}
    if labels:
        match_labels.update(labels)

    def labels_matcher(scenario):
        for label, value in match_labels.iteritems():
            if label not in scenario.labels:
                return False
            elif scenario.labels[label] != value:
                return False
        return (scenario.status in status)

    result = []
    for scenario in [c_scenario for c_scenario in context.scenarios_list if labels_matcher(c_scenario)]:
        intersection_hosts = list(set(checked_scenario.hosts_list).intersection(set(scenario.hosts_list)))
        log.debug('intersection: scnario: %s, ticket: %s, hosts: %s', checked_scenario.scenario_id, checked_scenario.ticket_key, intersection_hosts)
        if len(intersection_hosts) > 0:
            result.append(
                {
                    'scenario': scenario,
                    'hosts_ids': intersection_hosts,
                    'hosts_info': []
                }
            )
    return result


def get_hosts_info(context, ids):

    def get_host_additional_info(inv):
        try:
            data = context.walle_client.get_host(inv, fields=['location.switch', 'location.rack', 'project'])
        except WalleApiError as e:
            log.debug(e)
            return {}
        return {'switch': data['location']['switch'], 'w_prj': data['project'], 'rack': data['location']['rack']}

    def get_hosts_additional_info_iter(invs):
        try:
            data = context.walle_client.iter_hosts(invs=[int(i) for i in invs], fields=['location.switch', 'location.rack', 'project'])
        except WalleApiError as e:
            log.debug(e)
            data = {}
        for h in data:
            yield {'switch': h['location']['switch'], 'w_prj': h['project'], 'rack': h['location']['rack']}

    def get_w_prj_info(name):
        # add cache!!!
        result = {'w_prj_tags': [], 'w_prj_owners': []}
        data = context.walle_client.get_project(name, fields=["tags", "owners"])
        result['w_prj_tags'] = data.get('tags', [])
        result['w_prj_owners'] = data.get('owners', [])
        log.debug('Wall-e project {} info: tags={}, owners={}'.format(name, result['w_prj_tags'], result['w_prj_owners']))
        return result

    def get_abc_slug(abc_prj):
        # log.debug('Getting slug ABC project: {}'.format(abc_prj))
        # add cache!!!
        try:
            data = context.abc_client.service_name_to_slug(service_name=abc_prj)
            result = data
        except:  # noqa
            result = abc_prj
        return result

    def get_owners_from_abc(abc_prj_slug):
        result = []
        members = context.abc_client.get_members(service_slug=abc_prj_slug, roles=PRIORITY_OF_OWNERS_FROM_ABC)
        for i in PRIORITY_OF_OWNERS_FROM_ABC:
            if members.get(i):
                log.debug('Getting owners from {}: {}'.format(i, members.get(i)))
                result = members.get(i)
                break
        return result

    def calculate_owners_for_host(abc_prj_slug, w_prj_tags=[], w_prj_owners=[]):
        result = {'service_owners': [], 'hw_owners': []}
        for tag, owners in SERVICE_OWNERS_BY_W_TAG.items():
            if tag in w_prj_tags:
                log.debug('Getting owners from SERVICE_OWNERS_BY_W_TAG: {}'.format(owners))
                result['service_owners'].extend(owners)
        result['hw_owners'] = HW_OWNERS_BY_ABC.get(abc_prj_slug, [])
        if not result['hw_owners']:
            result['hw_owners'] = get_owners_from_abc(abc_prj_slug)
        if not result['hw_owners'] and w_prj_owners:
            for i in w_prj_owners:
                if i.startswith('@'):
                    pass
                    # TODO
                    # result['hw_owners'].add(context.staff_client.grp_to_logins(i))
                else:
                    result['hw_owners'].append(i)
        result['hw_owners'] = list(set([i for i in result['hw_owners'] if not i.startswith('robot-')]))[:5]
        return result

    host_info_none = {
        'dc': 'None',
        'abc_prj': 'None',
        'switch': 'None',
        'rack': 'None',
        'w_prj': 'None',
        'w_prj_tags': [],
        'w_prj_owners': [],
        'service_owners': [],
        'hw_owners': []}
    # BOT returns:
    # 'inv'
    # 'fqdn'
    # 'abc_prj'
    # 'dc'
    # 'rack'
    for id in ids:
        log.debug('Getting info for: {}'.format(id))
        if isinstance(id, int):
            id = str(id)
        host_info = context.hosts_info.get(id)
        if host_info:
            log.debug('Host info founded in cache: {}'.format(host_info))
            yield host_info
            continue
        host_info = host_info_none.copy()
        log.debug('Getting info for %s from BOT', id)
        if is_inv(id):
            try:
                host_info.update(context.bot_client.get_host_info(inv=id))
            except BotClientHostNotFound:
                host_info['inv'] = id
                log.debug('Host info not found: {}'.format(host_info))
                continue
        elif is_fqdn(id):
            try:
                host_info.update(context.bot_client.get_host_info(fqdn=id))
            except BotClientHostNotFound:
                host_info['fqdn'] = id
                log.debug('Host info not found: {}'.format(host_info))
                continue
        else:
            log.debug('Unrecognized ID %r', id)
            continue
        inv = host_info['inv']
        log.debug('Getting additional info for %r', inv)
        host_info.update(get_host_additional_info(inv))
        w_prj = host_info.get('w_prj', 'None')
        if w_prj != 'None':
            host_info.update(get_w_prj_info(w_prj))
        abc_prj_slug = get_abc_slug(host_info['abc_prj'])
        host_info['abc_prj'] = ABC_CNAME.get(abc_prj_slug, abc_prj_slug)
        # host_info.update(calculate_owners_for_host(host_info['abc_prj'], host_info['w_prj_tags'], host_info['w_prj_owners']))
        log.debug('Host info: {}'.format(host_info))
        context.hosts_info[host_info['inv']] = host_info
        context.hosts_info[host_info['fqdn']] = host_info
        yield host_info


def hosts_info_group_by_prj(context, hosts):
    """
    return: {
        abc_prj: {
            'hw_owners': [logins],
            'service_owners': [logins]
            'hosts': [data]
        }
    }
    """
    result = defaultdict(lambda: {'hw_owners': [], 'hosts': [], 'service_owners': []})
    for host in hosts:
        abc_prj = host['abc_prj']
        result[abc_prj]['hosts'].append(host)
        if not result[abc_prj]['service_owners']:
            result[abc_prj]['service_owners'] = host['service_owners']
        if not result[abc_prj]['hw_owners']:
            result[abc_prj]['hw_owners'] = list(map(lambda i: '{}'.format(i), host['hw_owners']))
    return result


def fetch_hosts_info_iter(context, hostlist, step=100, fields=None):
    for i in xrange(0, len(hostlist), step):
        j = min(i + step, len(hostlist))
        response = context.walle_client.get_hosts(
            fields=('state', 'status'),
            invs=hostlist[i:j]
            )
        yield response['result']


def get_ticket():
    pass


def get_tickets(ctx, scenarios, filter):
    def filter_tickets(tickets):
        scenarios_tickets = {s.ticket_key for s in scenarios}
        return [t for t in tickets if t.key not in scenarios_tickets]

    return filter_tickets(ctx.st_client.issues.find(filter))


def type_work_parser(block):
    type_works = {
        "ввод/перемещение хостов в wall-e проект": "add_hosts",
        "ввод хостов в wall-e проект": "add_hosts",
        "забрать хосты из предзаказа": "preorder_add_hosts",
        "вЫвод хостов из wall-e": "rm_hosts",
        "выключение хостов для ITDC": "power_off",
        "Hosts Power-OFF for ITDC maintenance": "power_off",
        "апгрейд хостов": "upgrade_hosts"
    }
    block_value = block[-1]
    return type_works.get(block_value, "unknown")


def is_fqdn(id):
    return fqdn_regex.match(id) is not None


def is_inv(id):
    return isinstance(id, int) or inv_regex.match(id) is not None


def id_parser(block):
    splitter = re.compile(r'[\s,;]+', flags=re.UNICODE)

    def is_valid_id(item):
        return is_fqdn(item) or is_inv(item)

    return [item for item in splitter.split('\n'.join(block)) if is_valid_id(item)]


def responsible_parser(block):
    pattern = re.compile('[^\(]\((.*)\)')
    for b in block:
        match = pattern.search(b)
        if match:
            return ascii_parser(match.group(1))
    return ''


def join_parser(block):
    '\n'.join(block)


def boolean_parser(block):
    return ascii_parser(block[-1]) in ('Yes', 'Да')


def ascii_parser(block):
    if type(block) == list:
        block = "\n".join(block)
    try:
        return block.encode('ascii')
    except Exception:
        return block


def ticket_parser(block):
    return ascii_parser(block).split('/')[-1]


def validate_parsed(data):
    if 'type' not in data.keys():
        raise InvalidParsedDataException("Empty data")
    if data.get('type') not in ('add_hosts', 'rm_hosts', 'power_off', 'preorder_add_hosts', 'upgrade_hosts', 'multi_dc_parent_processing'):
        raise InvalidParsedDataException("Invalid task type: '{}' in {}".format(data.get('type'),
                                                                                data.get('ticket_key')))
    if not data.get('hosts') and not data.get('type') == 'preorder_add_hosts':
        raise InvalidParsedDataException("Hosts list empty: in {}".format(data.get('ticket_key')))
    return data


def parse_ticket(ticket):
    log.info('Parsing %s', ticket.key)

    def parse_block(type):
        parsers = {
            'type': type_work_parser,
            'hosts': id_parser,
            'comment': ascii_parser,
            'responsible': responsible_parser,
            'dismantle': boolean_parser,
            'whole_preorder': boolean_parser,
            'ref_ticket_key': ticket_parser
        }
        if type in parsers:
            return parsers[type]
        else:
            return ascii_parser

    def split_main_block(lines):
        questions = {
            "Тип работ:": "type",
            "Operation type:": "type",
            "Список ID серверов:": "hosts",
            "List of host's IDs:": "hosts",
            "Требуется очистка:": "need_clean",
            "Демонтаж:": "dismantle",
            "Wall-e проект:": "target_project_id",
            "\"Wall-e проект из которого перемещаются хосты\" (опционально):": "source_project_id",
            "Комментарий:": "comment",
            "Кто сдает хосты:": "responsible",
            "Ответственный за работы:": "responsible",
            "Responsible for the maintenance:": "responsible",
            "Тикет из очереди ITDC:": "ref_ticket_key",
            "Ticket from ITDC queue:": "ref_ticket_key",
            "ID предзаказа:": "preorder_id",
            "Забрать все хосты из предзаказа:": "whole_preorder",
            "Переместить серверы в проект ABC после вывода:": "abc_service_name",
        }
        blocks = {}
        blockname = None
        for line in lines:
            line = line.encode('utf-8').strip()
            if line in questions.keys():
                blockname = questions[line]
                blocks[blockname] = []
                continue
            elif line and blockname:
                blocks[blockname].append(line)
        return blocks

    def get_main_block():
        lines = ticket.description.splitlines()
        start = 0
        i = 0
        for line in lines:
            if 'for_janitor' in line:
                start = i + 1
            if start and '}>' in line:
                return lines[start:i]
            i += 1

    main_block = get_main_block()
    parsed_args = {
        'ticket_key': str(ticket.key),
        'ticket_created_by': str(ticket.createdBy.login),
        'ticket_summary': ticket.summary,
        'ticket_creation_date': datetime.strftime(parser.parse(ticket.createdAt), '%Y-%m-%d')
        }
    if main_block:
        for k, v in split_main_block(main_block).items():
            parsed_args[k] = parse_block(k)(v)
    if MULTI_DC_TICKET_PARENT_TAG in ticket.tags:
        parsed_args['type'] = 'multi_dc_parent_processing'
    log.debug('Parsed data: {}'.format(parsed_args))
    return parsed_args


def get_parsed_ticket(ticket):
    return validate_parsed(parse_ticket(ticket))


def create_ticket(ctx, **kwargs):
    from infra.rtc.janitor.constants import ST_TICKET_QUEUE, ST_TICKET_TYPE
    import datetime
    kwargs['date'] = datetime.datetime.today().strftime('%d.%m.%Y')
    title = render_template('basic_header.jinja', **kwargs)
    description = render_template('basic_description.jinja',  **kwargs)
    tags = ['janitor_client']
    tags.extend(ST_TAGS_BY_SCENARIO_TYPE[kwargs['type']])
    new_issue = ctx.obj.st_client.issues.create(
        queue=ST_TICKET_QUEUE,
        type=ST_TICKET_TYPE,
        summary=title,
        description=description,
        tags=tags
    )
    return str(new_issue.key)


def st_create_ticket(context, **kwargs):
    log.debug(kwargs)
    new_issue = context.st_client.issues.create(
        queue=kwargs['queue'],
        type=kwargs.get('type'),
        tags=kwargs.get('tags'),
        summary=kwargs['summary'],
        description=kwargs.get('description'),
        links=kwargs.get('links')
    )
    return new_issue


def validate_scenario(f):
    def decors(cls, ctx, **kwargs):
        intersections = find_intersections_by_hosts(ctx.obj.walle_client, kwargs['hosts'])
        if intersections:
            for i in intersections:
                log.info('Found another scenarios with same hosts: %s', json.dumps(i, indent=4))
            message = render_template('hosts_interseption.jinja', intersect_scenarios=intersections)
            st_transistion(ctx.obj.st_client, kwargs['ticket_key'], state='closed', resolution='invalid', comment=message)
#            raise ScenarioHostIntesectionException("Another scenarios with same hosts exists")
        return f(cls, ctx, **kwargs)

    return decors


def validate_destination_project(f):
    def decors(cls, ctx, **kwargs):
        log.debug('validate_destination_project %s', kwargs)
        if 'target_project_id' in kwargs:
            try:
                ctx.obj.walle_client.get_project(id=kwargs['target_project_id'])
            except WalleApiError:
                message = render_template('project_not_exist.jinja', target_project_id=kwargs['target_project_id'])
                log.debug('arg %s,\nmsg %s', kwargs['target_project_id'], message)
                st_transistion(ctx.obj.st_client, kwargs['ticket_key'], state='closed', resolution='invalid', comment=message)
                log.debug('Destination project for scenario not exist: %s', kwargs['target_project_id'])
#                raise ScenarioProjectNotExistException("Destination project for scenario not exist")
        return f(cls, ctx, **kwargs)
    return decors


def validate_ticket(f):
    def decor(cls, ctx, **kwargs):
        if not kwargs['ticket_key']:
            ticket_key = create_ticket(ctx, **kwargs)
            log.info('Ticket %s created', ticket_key)
            kwargs['ticket_key'] = ticket_key
        return f(cls, ctx, **kwargs)
    return decor


def add_command_name(f):
    def decor(ctx, **kwargs):
        kwargs['type'] = ctx.command.name
        return f(ctx, **kwargs)
    return decor


def add_hosts_scenario_name(scenario_kwargs):
    if scenario_kwargs['labels']['task_name'] == 'add_hosts_qloud':
        target = 'Qloud'
    else:
        target = 'RTC'

    name = 'Add {number} hosts to {target} {project} ({ticket})'.format(
        target=target,
        number=len(scenario_kwargs.get('hosts', [])),
        project=scenario_kwargs['script_args']['target_project_id'],
        ticket=scenario_kwargs['ticket_key']
        )
    _SCENARIO_NAME_LIMIT = 100
    if len(name) > _SCENARIO_NAME_LIMIT:
        name = "Add {number} hosts to {target}".format(
            number=len(scenario_kwargs.get('hosts', [])),
            target=target
        )
    if len(name) > _SCENARIO_NAME_LIMIT:
        name = name[:_SCENARIO_NAME_LIMIT - 3] + "..."

    return name


def remove_hosts_scenario_name(scenario_kwargs):
    name = 'Remove {number} hosts from wall-e ({ticket})'.format(
        number=len(scenario_kwargs.get('hosts', [])),
        ticket=scenario_kwargs['ticket_key']
        )
    _SCENARIO_NAME_LIMIT = 100
    if len(name) > _SCENARIO_NAME_LIMIT:
        name = "Remove {number} hosts from wall-e".format(
            number=len(scenario_kwargs.get('hosts', [])),
        )
    if len(name) > _SCENARIO_NAME_LIMIT:
        name = name[:_SCENARIO_NAME_LIMIT - 3] + "..."

    return name


def drop_none(d, none=None):
    return {k: v for k, v in d.items() if v is not none}
