# -*- coding: utf-8 -*-

import logging

from .constants import (
    ITDC_POWEROFF_MAX_HOSTS,
    CHECK_HOSTS_PREREQUESTS_STAGE
)

from .common import register_stage, finish_stage, terminate_fsm
from infra.rtc.janitor.common import (
    render_template,
    add_st_comment,
    st_transistion,
    remove_st_janitor_processing,
)

from .misc import (
    filter_hosts_without_tag,
    filter_hosts_not_in_walle,
    filter_hosts_in_walle,
    list_hosts_racks,
    get_hosts_info
    )

log = logging.getLogger(__name__)


def check_hosts_prerequests_stage(context, scenario):
    if len(scenario.hosts_list) > ITDC_POWEROFF_MAX_HOSTS:
        message = render_template('power_off_message.jinja',
                                  text_case="ITDC_TOO_MANY_HOSTS",
                                  max_hosts_per_ticket=ITDC_POWEROFF_MAX_HOSTS,
                                  ticket_key=scenario.ticket_key,
                                  hosts_list=get_hosts_info(context, scenario.hosts_list)
                                  )
        st_transistion(context.st_client, scenario.ticket_key, state='closed', resolution='invalid', comment=message)
        try:
            add_st_comment(context.st_client, scenario.labels['ref_ticket_key'], text=message)
        except Exception as e:
            log.debug('Error while works with linked ticket: %r', e)
        remove_st_janitor_processing(context.st_client, scenario.ticket_key)
        return terminate_fsm(scenario)

    walle_hosts = filter_hosts_in_walle(context, scenario.hosts_list)
    walle_invs = [i.get('inv') for i in walle_hosts]
    racks = list_hosts_racks(context, scenario.hosts_list)
    walle_hosts_non_rtc = [int(h['inv']) for h in filter_hosts_without_tag('rtc', context, walle_invs)]
    not_walle_hosts = [int(h['inv']) for h in filter_hosts_not_in_walle(context, scenario.hosts_list)]
    invalid_hosts = walle_hosts_non_rtc + not_walle_hosts
    valid_hosts = [i for i in scenario.hosts_list if i not in invalid_hosts]
    log.debug('filtering lists: \n   walle_hosts_non_rtc: %r\n   not_walle_hosts: %r\n   invalid_hosts: %r\n   valid_hosts: %r\n   scenario.hosts_list: %r',
              walle_hosts_non_rtc,
              not_walle_hosts,
              invalid_hosts,
              valid_hosts,
              scenario.hosts_list)
    if len(racks) == 1:
        racks = []
    if any((not_walle_hosts, walle_hosts_non_rtc, racks)):
        message = render_template('power_off_message.jinja',
                                  text_case="ITDC_PREREQUESTS",
                                  ticket_key=scenario.ticket_key,
                                  hosts_list=get_hosts_info(context, scenario.hosts_list),
                                  walle_hosts=walle_hosts,
                                  walle_hosts_non_rtc=walle_hosts_non_rtc,
                                  not_walle_hosts=not_walle_hosts,
                                  hosts_racks=racks,
                                  valid_hosts=valid_hosts,
                                  )
        add_st_comment(context.st_client, scenario.ticket_key, text=message)
        try:
            add_st_comment(context.st_client, scenario.labels['ref_ticket_key'], text=message)
        except Exception as e:
            log.debug('Error while works with linked ticket: %r', e)
        log.debug(message)
        if not valid_hosts:
            message = render_template('power_off_message.jinja',
                                      text_case="ITDC_CLOSE",
                                      ticket_key=scenario.ticket_key,
                                      hosts_list=get_hosts_info(context, scenario.hosts_list),
                                      )
            st_transistion(context.st_client, scenario.ticket_key, state='closed', resolution='invalid',
                           comment=message)
            try:
                add_st_comment(context.st_client, scenario.labels['ref_ticket_key'], text=message)
            except Exception as e:
                log.debug('Error while works with linked ticket: %r', e)
            log.debug(message)
            remove_st_janitor_processing(context.st_client, scenario.ticket_key)
            return terminate_fsm(scenario)
        elif valid_hosts and not racks:
            message = render_template('power_off_message.jinja',
                                      text_case="ITDC_VALID_CONTINUE",
                                      ticket_key=scenario.ticket_key,
                                      hosts_list=get_hosts_info(context, scenario.hosts_list),
                                      valid_hosts=valid_hosts
                                      )
            add_st_comment(context.st_client, scenario.ticket_key, text=message)
            try:
                add_st_comment(context.st_client, scenario.labels['ref_ticket_key'], text=message)
            except Exception as e:
                log.debug('Error while works with linked ticket: %r', e)
            log.debug('List of valid hosts %r', valid_hosts)
            log.debug(message)
            scenario.hosts = [h for h in scenario.hosts if h['inv'] in valid_hosts]
            log.debug('new scenario host list: %r', scenario.hosts)
            scenario.save(context.walle_client)
    st_transistion(context.st_client, scenario.ticket_key, state='readyForDev')
    return finish_stage(scenario)


register_stage(CHECK_HOSTS_PREREQUESTS_STAGE, check_hosts_prerequests_stage)
