# coding: utf-8

import logging
from collections import namedtuple

from .constants import (
    ACTION_DONE,
    ACTION_PASS,
    ACTION_RETRY,
    ADD_COMMENT_FOR_DUTY_TO_ST_STAGE,
    ADD_TAGS_TO_ST_STAGE,
    CHECK_PERMS_FOR_HOSTS_STAGE,
    CHECK_TASK_COUNT_STAGE,
    CHECK_TO_MNT_COMPLETE,
    CHECK_HOSTS_PREREQUESTS_STAGE,
    CREATE_SCENARIO_STAGE,
    VALIDATE_DESTINATION_PROJECT_STAGE,
    VALIDATE_TICKET_STAGE,
    VALIDATE_SCENARIO_STAGE,
    RESOLVE_HOSTS_STAGE,
    START_SCENARIO_STAGE,
    CHECK_RESPONSIBLE_MEMBERSHP_STAGE,
    CHECK_INTERSEPT_POWEROFF_STAGE,
    CHECK_INTERSEPT_STARTED_STAGE,
    SET_DUTY_AS_ASSIGNEE,
    SET_DUTY_AS_ASSIGNEE_UPGRADES,
    PROCESS_MULTI_DC_TASK_STAGE,
    ADD_PREORDER_STAGE,
    REMOVE_PROCESSING_TAG,
    PROCESS_MULTIDC_TICKET,
    RESOLVE_ABC_ID_STAGE,
)

log = logging.getLogger(__name__)

_STAGES = {}
_StageConfig = namedtuple("StageConfig", ["handler"])


def register_stage(name, handler):
    assert "," not in name, "wrong name specified, no comma allowed"
    stage_config = _StageConfig(handler)
    if _STAGES.setdefault(name, stage_config) is not stage_config:
        raise RuntimeError("Unable to register '{}' stage: it already exists.".format(name))
    else:
        log.debug("Register handle %r as %s", handler, name)


def _get_stage_handler(scenario):
    fsm_stages = scenario.fsm_stages
    if not fsm_stages:
        raise RuntimeError("Can't handle scenario with no stages")
    stage_name = fsm_stages[-1]
    stage_config = _STAGES.get(stage_name)
    if stage_config is None:
        raise RuntimeError("Stage {} not registered".format(stage_name))
    return stage_config.handler


def finish_stage(scenario):
    fsm_stages = scenario.fsm_stages
    if not fsm_stages:
        raise RuntimeError("Can't finish scenario with no stages")
    scenario.fsm_prev_stage = fsm_stages[-1]
    scenario.fsm_stages = fsm_stages[:-1]
    return ACTION_DONE


def terminate_fsm(scenario):
    scenario.fsm_prev_stage = scenario.fsm_curr_stage
    scenario.fsm_stages = []
    return ACTION_DONE


def retry_stage(scenario):
    fsm_stages = scenario.fsm_stages
    if not fsm_stages:
        raise RuntimeError("Can't retry scenario with no stages")
    scenario.fsm_prev_stage = fsm_stages[-1]
    return ACTION_RETRY


def append_stage(client, scenario, *args):
    fsm_stages = scenario.fsm_stages
    fsm_stages.extend(args)
    log.debug('Adding new stages: {}'.format(args))
    scenario.fsm_stages = fsm_stages
    scenario.save(client)


def set_stage(client, scenario, *args):
    scenario.fsm_prev_stage = scenario.fsm_curr_stage
    fsm_stages = args
    log.debug('Set new stages: {}'.format(args))
    scenario.fsm_stages = fsm_stages
    # scenario.save(client)


def handle_new_scenario(client, scenario):
    task_name = scenario.labels['task_name']
    if task_name == 'power_off':
        stages = (
            # NOOP_STAGE,
            CHECK_TO_MNT_COMPLETE,
            START_SCENARIO_STAGE,
            CHECK_TASK_COUNT_STAGE,
            CHECK_INTERSEPT_STARTED_STAGE,
            REMOVE_PROCESSING_TAG,
            CREATE_SCENARIO_STAGE,
            CHECK_HOSTS_PREREQUESTS_STAGE,
            CHECK_INTERSEPT_POWEROFF_STAGE,
            SET_DUTY_AS_ASSIGNEE,
            RESOLVE_HOSTS_STAGE,
            CHECK_RESPONSIBLE_MEMBERSHP_STAGE,
        )
    elif task_name in {'add_hosts', 'add_hosts_qloud'}:
        stages = (
            REMOVE_PROCESSING_TAG,
            START_SCENARIO_STAGE,
            ADD_TAGS_TO_ST_STAGE,
            CREATE_SCENARIO_STAGE,
            VALIDATE_DESTINATION_PROJECT_STAGE,
            PROCESS_MULTI_DC_TASK_STAGE,
            RESOLVE_HOSTS_STAGE,
            VALIDATE_TICKET_STAGE,
            VALIDATE_SCENARIO_STAGE
        )
    elif task_name in {'rm_hosts'}:
        stages = (
            REMOVE_PROCESSING_TAG,
            START_SCENARIO_STAGE,
            ADD_TAGS_TO_ST_STAGE,
            CREATE_SCENARIO_STAGE,
            RESOLVE_ABC_ID_STAGE,
            RESOLVE_HOSTS_STAGE,
            VALIDATE_TICKET_STAGE,
            VALIDATE_SCENARIO_STAGE
        )
    elif task_name == 'preorder_add_hosts':
        stages = (
            REMOVE_PROCESSING_TAG,
            ADD_PREORDER_STAGE,
            SET_DUTY_AS_ASSIGNEE,
            VALIDATE_DESTINATION_PROJECT_STAGE
        )
    elif task_name == 'upgrade_hosts':
        stages = (
            REMOVE_PROCESSING_TAG,
            START_SCENARIO_STAGE,
            SET_DUTY_AS_ASSIGNEE_UPGRADES
        )
    elif task_name == 'multi_dc_parent':
        stages = (
            REMOVE_PROCESSING_TAG,
            PROCESS_MULTIDC_TICKET,
        )
    else:
        stages = (
            ADD_COMMENT_FOR_DUTY_TO_ST_STAGE,
            CHECK_PERMS_FOR_HOSTS_STAGE,
            ADD_TAGS_TO_ST_STAGE,
            REMOVE_PROCESSING_TAG,
            CREATE_SCENARIO_STAGE,
            RESOLVE_HOSTS_STAGE,
            VALIDATE_DESTINATION_PROJECT_STAGE,
            VALIDATE_TICKET_STAGE,
            VALIDATE_SCENARIO_STAGE
        )
    append_stage(client, scenario, *stages)


def handle_scenario_stages(context, scenario):
    if scenario.fsm_processed:
        raise RuntimeError("{!r} already processed".format(scenario))
    finished = False
    for _ in xrange(64):
        handler = _get_stage_handler(scenario)
        log.debug("Applying %r for %s", handler, scenario.name)
        action = handler(context, scenario)
        if action is ACTION_DONE:
            log.debug("Handler %r for %s exited successfully", handler, scenario.name)
            if not scenario.fsm_stages:
                log.info("Scenario %s is finished", scenario.name)
                scenario.mark_as_processed()
                finished = True
            log.debug("Saving scenario %s", scenario.name)
            scenario.save(context.walle_client)
        elif action is ACTION_RETRY:
            log.debug("Handler %r for %s not ready yet, retrying", handler, scenario.name)
            log.debug("Saving scenario %s", scenario.name)
            scenario.save(context.walle_client)
            finished = True
        elif action is ACTION_PASS:
            finished = True
        else:
            raise RuntimeError("Wrong action returned by {!r} for {!r}".format(handler, scenario))

        if finished:
            return

    raise RuntimeError("{!r} fell into to busy loop".format(scenario))
