# coding: utf-8

import logging
from collections import defaultdict
from .constants import (
    PROCESS_MULTI_DC_TASK_STAGE,
)
from .common import register_stage, finish_stage, terminate_fsm
from infra.rtc.janitor.common import (
    render_template,
    get_hosts_info,
    st_create_ticket,
    st_get_linked_multidc_subtickets,
    add_st_tags,
    st_transistion
)
from infra.rtc.janitor.constants import (
    MULTI_DC_MARKER,
    MULTI_DC_TICKET_SUBTICKET_TAG,
    MULTI_DC_TICKET_SUBTICKET_DC_TAG_PREFIX,
    MULTI_DC_TICKET_PARENT_TAG,
    MULTI_DC_TICKET_RELATIONSHIP,
    JANITOR_PROCESSING_TAG
)

log = logging.getLogger(__name__)


def create_subticket_for_add_hosts_limited_by_dc(context, dc, hosts_in_dc, target_project, original_ticket, responsible):
    title = render_template(
        'multi_dc_subticket.jinja',
        text_case="TITLE",
        DC=dc,
        dc_destonation_project=target_project,
        main_ticket=original_ticket.key)
    description = render_template(
        'multi_dc_subticket.jinja',
        text_case="BODY",
        DC=dc,
        dc_destonation_project=target_project,
        hosts=hosts_in_dc,
        responsible=responsible
        )
    tags = set(original_ticket.tags)
    tags.difference_update((MULTI_DC_TICKET_SUBTICKET_TAG, MULTI_DC_TICKET_PARENT_TAG))
    tags.update((MULTI_DC_TICKET_SUBTICKET_TAG, JANITOR_PROCESSING_TAG, '{}:{}'.format(MULTI_DC_TICKET_SUBTICKET_DC_TAG_PREFIX, dc)))
    log.debug('creating subticket')
    new_issue = st_create_ticket(
        context,
        queue=original_ticket.queue.key,
        type=original_ticket.type.key,
        tags=tuple(tags),
        summary=title,
        description=description,
        links=[{
            "relationship": MULTI_DC_TICKET_RELATIONSHIP,
            "issue": original_ticket.key
        }]
    )
    log.debug('create ticket: %r %s,  DC: %s, project: %s, hosts: %s', new_issue.key, title, dc, target_project, str(len(hosts_in_dc)))


def get_processed_dc(context, ticket_key):
    res = []
    for subticket in st_get_linked_multidc_subtickets(context.st_client, ticket_key, MULTI_DC_TICKET_SUBTICKET_TAG):
        for tag in subticket['ticket'].tags:
            if tag.startswith(MULTI_DC_TICKET_SUBTICKET_DC_TAG_PREFIX):
                res.append(tag.split(':')[1])
    return res


def process_multi_dc_task_stage(context, scenario):
    if MULTI_DC_MARKER not in scenario.script_args['target_project_id']:
        return finish_stage(scenario)
    subtickets_project_template = scenario.script_args['target_project_id'].replace('[dc]', '{}')
    log.debug('ticket: %s - multi-DC scenario, template: %s', scenario.ticket_key, subtickets_project_template)
    processed_dc = get_processed_dc(context, scenario.ticket_key)
    log.debug('ticket: %s, processed DCs %r', scenario.ticket_key, processed_dc)
    original_ticket = context.st_client.issues[scenario.ticket_key]
    hosts_by_dc = defaultdict(list)
    for host in get_hosts_info(context, scenario.hosts_list):
        hosts_by_dc[host['dc']].append(host['inv'])
    log.debug('founded DC`s: %s', hosts_by_dc.keys())
    for dc, dc_hosts in hosts_by_dc.iteritems():
        if dc not in processed_dc:
            subticket_project = subtickets_project_template.format(dc)
            create_subticket_for_add_hosts_limited_by_dc(context, dc, dc_hosts, subticket_project, original_ticket, scenario.labels['responsible'])
    log.debug('ticket: %s - add tags: %s', scenario.ticket_key, [MULTI_DC_TICKET_PARENT_TAG])
    add_st_tags(context.st_client, original_ticket.key, [MULTI_DC_TICKET_PARENT_TAG])
    st_transistion(context.st_client, original_ticket.key, state='readyForDev')
    st_transistion(context.st_client, original_ticket.key, state='inProgress')
    return terminate_fsm(scenario)


register_stage(PROCESS_MULTI_DC_TASK_STAGE, process_multi_dc_task_stage)
