# coding: utf-8

# from __future__ import print_function
# import sys
import time
import logging
from collections import defaultdict
import pprint

from infra.rtc.janitor.common import add_hosts_scenario_name, remove_hosts_scenario_name, drop_none
from infra.rtc.janitor.constants import (
    PROG_NAME,
    PROG_VER,
    LABEL_FSM_STAGES,
    LABEL_FSM_PROCESSED,
    LABEL_FSM_PREV_STAGE
)

# from .common import validate_scenario, validate_ticket, validate_destination_project, validate_hosts

log = logging.getLogger(__name__)


class Scenario(object):

    default_scenario_labels = {
        'source': PROG_NAME,
        'source_ver': PROG_VER}

    def __init__(self, creation_time, scenario_id, hosts, issuer, labels, name, scenario_type, status, ticket_key=None, script_args=None, action_time=None, **kwargs):
        self.scenario_id = scenario_id
        self.hosts = hosts
        self.issuer = issuer
        self.labels = labels
        self.name = name
        self.scenario_type = scenario_type
        self.script_args = script_args
        self.status = status
        self.ticket_key = ticket_key
        self.creation_time = creation_time
        if not action_time:
            self.action_time = creation_time
        else:
            self.action_time = action_time

    @property
    def fsm_stages(self):
        stages = self.labels.get(LABEL_FSM_STAGES)
        if not stages:
            return []
        return stages.split(",")

    @fsm_stages.setter
    def fsm_stages(self, stages):
        self.labels[LABEL_FSM_STAGES] = ",".join(stages)

    @property
    def hosts_list(self):
        return [i['inv'] for i in self.hosts]

    @hosts_list.setter
    def hosts_list(self, invs):
        if self.status == 'none':
            self.hosts = [{u'status': u'none', u'timestamp': self.creation_time, u'inv': h} for h in invs]

    @property
    def fsm_prev_stage(self):
        return self.labels.get(LABEL_FSM_PREV_STAGE, '')

    @fsm_prev_stage.setter
    def fsm_prev_stage(self, stage):
        self.labels[LABEL_FSM_PREV_STAGE] = stage

    @property
    def fsm_curr_stage(self):
        return self.labels.get(LABEL_FSM_STAGES, '').split(",")[-1]

    @property
    def fsm_processed(self):
        return LABEL_FSM_PROCESSED in self.labels

    def mark_as_processed(self):
        self.labels[LABEL_FSM_PROCESSED] = 'yes'

    def to_dict(self):
        return dict(vars(self))

    @classmethod
    def add_hosts(cls, client, **kwargs):
        data = defaultdict()
        data['labels'] = dict(cls.default_scenario_labels)
        data['labels'].update({
            'task_name': 'add_hosts',
            'ticket_created_by': kwargs['ticket_created_by'],
            'responsible': kwargs.get('responsible'),
            'comment': kwargs.get('comment', '')})
        data['scenario_type'] = 'hosts-transfer'
        data['ticket_key'] = kwargs['ticket_key']
        data['script_args'] = {
            'target_project_id': kwargs['target_project_id'],
            }
        data['hosts'] = kwargs['hosts']
        data['name'] = add_hosts_scenario_name(data)

        return cls._add_scenario(client, **data)

    @classmethod
    def add_hosts_qloud(cls, client, **kwargs):
        data = defaultdict()
        data['labels'] = dict(cls.default_scenario_labels)
        data['labels'].update({
            'task_name': 'add_hosts_qloud',
            'ticket_created_by': kwargs['ticket_created_by'],
            'responsible': kwargs.get('responsible'),
            'comment': kwargs.get('comment', ''),
            })
        data['scenario_type'] = 'hosts-add-qloud'
        data['ticket_key'] = kwargs['ticket_key']
        data['script_args'] = {
            'schedule_type': 'all',
            'target_hardware_segment': kwargs['target_hardware_segment']
            }
        data['hosts'] = kwargs['hosts']
        data['name'] = add_hosts_scenario_name(data)
        return cls._add_scenario(client, **data)

    @classmethod
    def move_hosts(cls, client, **kwargs):
        data = defaultdict()
        data['labels'] = dict(cls.default_scenario_labels)
        data['labels'].update({
            'task_name': 'move_hosts',
            'ticket_created_by': kwargs['ticket_created_by'],
            'responsible': kwargs.get('responsible'),
            'comment': kwargs.get('comment', '')})
        data['scenario_type'] = 'wait'
        data['ticket_key'] = kwargs['ticket_key']
        data['script_args'] = {
            'schedule_type': 'all',
            'target_project_id': kwargs['target_project_id'],
            }
        data['hosts'] = kwargs['hosts']
        # TODO: fix length limit in wall-e
        # scenario_kwargs['name'] = '{} add/move hosts to {}'.format(kwargs['ticket_key'], kwargs['target_project_id'])
        data['name'] = str(time.time())
        return cls._add_scenario(client, **data)

    @classmethod
    def rm_hosts(cls, client, **kwargs):
        data = defaultdict()
        data['labels'] = dict(cls.default_scenario_labels)
        data['labels'].update(drop_none({
            'task_name': 'rm_hosts',
            'comment': kwargs.get('comment', ''),
            'ticket_created_by': kwargs['ticket_created_by'],
            'responsible': kwargs.get('responsible'),
            'dismantle': str(kwargs.get('dismantle', False)),
            'abc_service_name': kwargs.get('abc_service_name'),
        }))
        data['scenario_type'] = 'hosts-transfer'
        data['ticket_key'] = kwargs['ticket_key']
        data['script_args'] = {
            'delete': True,
            }
        data['hosts'] = kwargs['hosts']
        data['name'] = remove_hosts_scenario_name(data)
        log.debug(data)
        return cls._add_scenario(client, **data)

    @classmethod
    def power_off(cls, client, **kwargs):
        data = defaultdict()
        data['labels'] = dict(cls.default_scenario_labels)
        data['labels'].update({
            'task_name': 'power_off',
            'ref_ticket_key': kwargs['ref_ticket_key'],
            'ticket_created_by': kwargs['ticket_created_by'],
            'responsible': kwargs['responsible'],
            'comment': kwargs.get('comment', '')
            })
        data['scenario_type'] = 'switch-to-maintenance'
        data['ticket_key'] = kwargs['ticket_key']
        data['script_args'] = {
            'schedule_type': 'all',
            }
        data['hosts'] = kwargs['hosts']
        # TODO: fix length limit in wall-e
        # scenario_kwargs['name'] = '{} add/move hosts to {}'.format(kwargs['ticket_key'], kwargs['target_project_id'])
        data['name'] = '{}_poweroff'.format(kwargs['ticket_key'])
        return cls._add_scenario(client, **data)

    @classmethod
    def preorder_add_hosts(cls, client, **kwargs):
        data = {}
        data['labels'] = dict(cls.default_scenario_labels)
        data['labels'].update({
            'task_name': 'preorder_add_hosts',
            'ticket_created_by': kwargs['ticket_created_by']
        })
        data['scenario_type'] = 'preorder-add-hosts'
        data['ticket_key'] = kwargs['ticket_key']
        data['script_args'] = {
            'target_project_id': kwargs['target_project_id'],
            'preorder_id': kwargs['preorder_id'],
            'whole_preorder': kwargs['whole_preorder']
        }
        data['name'] = '{}_preorder_{}_to_{}'.format(kwargs['ticket_key'], kwargs['preorder_id'], kwargs['target_project_id'])
        return cls._add_scenario(client, **data)

    @classmethod
    def upgrade_hosts(cls, client, **kwargs):
        data = defaultdict()
        data['labels'] = dict(cls.default_scenario_labels)
        data['labels'].update({
            'task_name': 'upgrade_hosts',
            'comment': kwargs.get('comment', ''),
            'ticket_created_by': kwargs['ticket_created_by'],
            'responsible': kwargs.get('responsible')
        })
        data['scenario_type'] = 'noop'
        data['ticket_key'] = kwargs['ticket_key']
        data['script_args'] = {
            'schedule_type': 'all',
            }
        data['hosts'] = kwargs['hosts']
        data['name'] = '{}_upgrade'.format(kwargs['ticket_key'])
        log.debug(data)
        return cls._add_scenario(client, **data)

    @classmethod
    def multi_dc_parent_processing(cls, client, **kwargs):
        data = defaultdict()
        data['labels'] = dict(cls.default_scenario_labels)
        data['labels'].update({
            'task_name': 'multi_dc_parent',
            'comment': kwargs.get('comment', ''),
            'ticket_created_by': kwargs['ticket_created_by'],
            'responsible': kwargs.get('responsible')
        })
        data['scenario_type'] = 'noop'
        data['ticket_key'] = kwargs['ticket_key']
        data['script_args'] = {
            'schedule_type': 'all',
            }
        data['hosts'] = kwargs['hosts']
        data['name'] = '{}_multi_dc_parent'.format(kwargs['ticket_key'])
        log.debug(data)
        return cls._add_scenario(client, **data)

    @classmethod
    def _add_scenario(cls, client, **data):
        ts = time.time()
        data['hosts'] = [{u'status': u'none', u'timestamp': ts, u'inv': h} for h in data.get('hosts', [])]
        return cls(
            creation_time=ts,
            scenario_id=None,
            status='none',
            issuer='none',
            **data)

    def create_walle_scenario(self, client):
        if self.scenario_id:
            log.warning("Scenario %s already exists: %s %s", self.scenario_id, self.scenario_type, self.name)
            return

        passed_keys = {'scenario_type', 'ticket_key', 'name', 'labels', 'hosts', 'script_args'}
        passed_data = {k: v for k, v in self.to_dict().iteritems() if k in passed_keys}
        invs = [i['inv'] for i in passed_data['hosts']]
        passed_data['hosts'] = invs
        log.debug('Create scenario in Wall-E: %r', passed_data)
        try:
            response = client.add_scenario(**passed_data)
        except Exception as e:
            log.error('Error while create scenario in Wall-E with\n --- params:\n%s,\n --- error:\%s', pprint.pformat(passed_data), e)
            raise
        else:
            log.debug(response)
            self.scenario_id = response['scenario_id']
            self.scenario_type = response['scenario_type']
            self.status = response['status']
            self.ticket_key = response['ticket_key']
            self.name = response['name']
            self.labels = response['labels']
            self.hosts = response['hosts']
            self.status = response['status']
            self.issuer = response['issuer']
            self.creation_time = response['creation_time']
            self.action_time = response.get('action_time', self.creation_time)
        pass

    @classmethod
    def load_all(cls, client, **kwargs):
        for scenario_kwargs in client.iter_scenarios(**kwargs):
            try:
                yield cls(**scenario_kwargs)
                log.debug('Scenario %s loaded successfully', scenario_kwargs['scenario_id'])
            except Exception:
                log.exception('Error loading scenario %s with params %s',
                              scenario_kwargs['scenario_id'], scenario_kwargs)

    @classmethod
    # @validate_scenario
    # @validate_ticket
    # @validate_destination_project
    # @validate_hosts
    def action(cls, ctx, **kwargs):
        log.info('Trying to create scenario with args: {}'.format(kwargs))
        method = getattr(cls, kwargs['type'])
        return method(ctx.obj.walle_client, **kwargs)

    def save(self, client):
        if not self.scenario_id:
            return
        else:
            # NOTE(rocco66): 'WORK_STATUS' is 'system' walle label, do not try to update it
            self.labels.pop('WORK_STATUS', None)
            data = {
                'scenario_id': self.scenario_id,
                'labels': self.labels
                }
        log.debug('Modify scerario: %r', data)
        response = client.modify_scenario(**data)
        log.debug(response)
        self.scenario_type = response['scenario_type']
        self.status = response['status']
        self.ticket_key = response['ticket_key']
        self.name = response['name']
        self.labels = response['labels']
        self.hosts = response['hosts']
        self.status = response['status']
        self.issuer = response['issuer']
        self.creation_time = response['creation_time']
        self.action_time = response.get('action_time', self.creation_time)

    def start(self, client):
        if not self.scenario_id:
            raise Exception("Starting scenario before creating it in Wall-E")
        response = client.start_scenario(self.scenario_id, self.ticket_key)
        log.debug(response)
        self.scenario_type = response['scenario_type']
        self.status = response['status']
        self.ticket_key = response['ticket_key']
        self.name = response['name']
        self.labels = response['labels']
        self.hosts = response['hosts']
        self.status = response['status']
        self.issuer = response['issuer']
        self.creation_time = response['creation_time']
        self.action_time = response.get('action_time', self.creation_time)
        log.debug('Scenario %s #%r started', self.name, self.scenario_id)

    def __repr__(self):
        return "<Scenario {!r}>".format(dict(vars(self)))
