import os
import subprocess
import json
import io

from juggler.bundles import as_check, Status, Event

name = 'eth_errors'
state_file = '/dev/shm/{}_state'.format(name)


def get_devices():
    rv = []
    for fname in os.listdir('/sys/class/net/'):
        if fname.startswith('eth'):
            rv.append(fname)
    return rv


def driver_check(interface):
    res = subprocess.check_output(['/sbin/ethtool', '-i', interface])
    for line in res.split(b'\n'):
        if not line.startswith(b'driver'):
            continue
        driver_name = line.split()[1]
        if driver_name.startswith(b'mlx') or driver_name.startswith(b'ixgbe'):
            return True
    return False


def check_errors(interface):
    res = subprocess.check_output(['/sbin/ethtool', '-S', interface])
    errors = {
        'rx_errors': 0,
        'tx_errors': 0,
        'rx_dropped': 0,
        'tx_dropped': 0,
        'rx_length_errors': 0,
        'rx_over_errors': 0,
        'rx_crc_errors': 0,
        'rx_frame_errors': 0,
        'rx_fifo_errors': 0,
        'rx_missed_errors': 0,
        'tx_aborted_errors': 0,
        'tx_carrier_errors': 0,
        'tx_fifo_errors': 0,
        'tx_heartbeat_errors': 0,
        'tx_window_errors': 0
    }
    state = {'packets': {}, 'bytes': {}}
    for q in range(64):
        state['packets']['tx{}_packets'.format(q)] = 0
        state['bytes']['tx{}_bytes'.format(q)] = 0
    for line in io.BytesIO(res):
        line = line.strip()
        # lines look like: tx_pause_storm_warning_events : 0
        pos = line.find(b':')
        if pos == -1:
            continue
        name = line[:pos].strip().decode("utf-8")
        count = line[pos+1:].strip()
        if name in errors:
            errors[name] = int(count)
        if name in state['packets']:
            state['packets'][name] = int(count)
        if name in state['bytes']:
            state['bytes'][name] = int(count)
    return errors


def get_prev_state(interface):
    try:
        with open(state_file + '_' + interface, 'r') as f:
            return json.load(f)
    except (ValueError, IOError):
        return None


def save_state(interface, state):
    try:
        with open(state_file + '_' + interface, 'w') as f:
            json.dump(state, f)
    except Exception as e:
        return "Failed to save state: {}".format(e)
    else:
        return None


@as_check(name=name)
def juggler_check():
    parts = []
    for dev in get_devices():
        if driver_check(dev):
            errors = check_errors(dev)
            prev_errors = get_prev_state(dev)
            err = save_state(dev, errors)
            if err is not None:
                return Event(Status.CRIT, err)
            if prev_errors is None:
                continue
            for errname, count in errors.items():
                delta = count - prev_errors.get(errname, 0)
                if delta != 0:
                    parts.append("{} for dev {} delta is {}, absolute is {}".format(errname, dev, delta, count))

    if parts:
        return Event(Status.CRIT, "; ".join(parts))
    else:
        return Event(Status.OK, "Ok")
