"""
file - file/directory status check

Files should be defined using check opts, see
https://wiki.yandex-team.ru/sm/juggler/juggler-client/#parametrizacijaproveroknastoroneservera

Reinventing `test` util (man 1 test), to avoid arbitrary code execution (which
is highly possible when `test` called as subprocess).

"""
import argparse
import os.path

from juggler.bundles import as_check, Status, Event


CHECK_NAME = 'is_file'


class CheckArgParserError(Exception):
    pass


class CheckArgParser(argparse.ArgumentParser):
    """ Override method to prevent exiting for correct error reporting """
    def error(self, message=None):
        raise CheckArgParserError(message)


@as_check(name=CHECK_NAME)
def juggler_check(*args):
    status_map = Status.__members__
    status_keys = status_map.keys()

    argparser = CheckArgParser()
    argparser.add_argument('--false-message', type=str, default='False')
    argparser.add_argument('--false-status', type=str, choices=status_keys,
                           default='CRIT')
    argparser.add_argument('--true-message', type=str, default='True')
    argparser.add_argument('--true-status', type=str, choices=status_keys,
                           default='CRIT')
    argparser.add_argument('file', type=str, help='file to check')

    single = argparser.add_mutually_exclusive_group(required=True)
    single.add_argument('-e', '--exists', action='store_true')

    try:
        opts = argparser.parse_args(args=args)
    except CheckArgParserError as e:
        return Event(Status.CRIT, str(e))

    try:
        if opts.exists:
            cond = os.path.exists(opts.file)
        else:
            return Event(Status.CRIT, 'Unsupported mode')
    except BaseException as e:
        return Event(Status.CRIT, str(e))

    if cond:
        return Event(status_map[opts.true_status], opts.true_message)

    return Event(status_map[opts.false_status], opts.false_message)
