"""
NTP check
"""

from juggler.bundles import as_check, Status
from infra.rtc.juggler.bundle.util import WalleCompatEvent
from subprocess import run  # PIPE, DEVNULL, Popen, run
import re
import os

CHECK_NAME = 'ntp'
WARM_UP_UPTIME_SEC = 15 * 60

associations_delimeter = re.compile(r'\s+')
sysdata_splitter = re.compile(r'^(?P<key>[^:]+): +(?P<val>.*)$')

OK = Status.OK.value
WARN = Status.WARN.value
CRIT = Status.CRIT.value

ntpd_status = {}
ntpd_associations = {}


def collect_ntp_data(commands):
    returned_data = {}
    for cmd in commands:
        cmd_result = run(['/usr/bin/ntpq', '-n', '-c', cmd], capture_output=True, text=True, universal_newlines=True,
                         timeout=60)
        cmd_result.check_returncode()
        if os.environ.get('COMPATIBILITY_MODE', 'false') == 'true':
            # Compatibility hack for precise/trusty ntpd, see LOGBROKER-6931 for details
            # ntp check is used by logbroker team outside RTC
            # Please notify someone from logbroker team in case of breaking changes
            if cmd_result.stderr.strip() == '''***Command `{}' unknown'''.format(cmd):
                # ntpq on trusty doesn't accept 'sysstat' command, so ntpdc used
                cmd_result = run(
                    ['/usr/bin/ntpdc', '-n', '-c', cmd], capture_output=True, text=True, universal_newlines=True, timeout=60
                )
        if len(cmd_result.stdout) == 0:
            raise Exception('Can not communicate with NTPD service')
        returned_data[cmd] = cmd_result.stdout.strip()
    return returned_data


def parse_rv_output(data):
    # Parse rv command output and add key=value pairs to ntpd_status
    key = None
    val = []
    parsed_data = {}
    for string in data.split():
        if '=' in string:
            if key is not None:
                val = ' '.join(val)
                val = val.rstrip(',').strip('"')
                parsed_data[key] = val
            expression = string.split('=')
            key = expression[0]
            val = [expression[1]]
        else:
            val.append(string)
    return parsed_data


def parse_as_output(data):
    # Parse as output and fill associations (ntp clients and servers) to dict
    parsed_data = {}
    associations_lines = data.splitlines()
    for line in associations_lines[2:]:
        associations_list = associations_delimeter.split(line)
        id_ = associations_list[2]
        parsed_data[id_] = {
            'status': associations_list[3],
            'configured': True if associations_list[4] == 'yes' else False,
            'reachable': True if associations_list[5] == 'yes' else False,
            'condition': associations_list[7]
        }
    return parsed_data


def parse_sys_output(data):
    # Parse sysstat and sysinfo command output and add key=value pairs to ntpd_status
    parsed_data = {}
    for line in data.splitlines():
        parsed = sysdata_splitter.match(line.strip())
        if parsed is not None:
            parsed_data[parsed.group('key')] = parsed.group('val')
    return parsed_data


def make_decision(status, associations):
    j_status = [OK]
    msg = []
    # uptime
    ntp_uptime = min(
        (int(status[option]) for option in ('time since restart', 'time since reset', 'uptime', 'sysstats reset') if
         option in status))
    if ntp_uptime > WARM_UP_UPTIME_SEC:
        # stratum
        if int(status['stratum']) > 4:
            j_status.append(CRIT)
            msg.append('stratum is too high')
        # associations: sys.peer & 2 x candidates & no one reject or falsetick
        if not any((association['status'][1] == '6' for association in associations.values())):  # status == sys.peer
            j_status.append(CRIT)
            msg.append('no primary ntp sync server')
        if sum((1 for association in associations.values() if
                association['status'][1] == '4')) < 2:  # status == candidate
            j_status.append(WARN)
            msg.append('less than 2 candidates to sync server')
        if any((association['status'][1] in ('0', '1') for association in
                associations.values())):  # status == reject or falsetick
            j_status.append(WARN)
            msg.append('invalid peers detected')
        if not all((association['reachable'] for association in associations.values())):
            j_status.append(WARN)
            msg.append('some peers unreachable')
        # NTPd "status -> source" decoding
        # Code 	Message 	Description
        # 0 	sync_unspec 	not yet synchronized
        # 1 	sync_pps 	pulse-per-second signal (Cs, Ru, GPS, etc.)
        # 2 	sync_lf_radio 	VLF/LF radio (WWVB, DCF77, etc.)
        # 3 	sync_hf_radio 	MF/HF radio (WWV, etc.)
        # 4 	sync_uhf_radio 	VHF/UHF radio/satellite (GPS, Galileo, etc.)
        # 5 	sync_local 	local timecode (IRIG, LOCAL driver, etc.)
        # 6 	sync_ntp 	NTP
        # 7 	sync_other 	other (IEEE 1588, openntp, crony, etc.)
        # 8 	sync_wristwatch 	eyeball and wristwatch
        # 9 	sync_telephone 	telephone modem (ACTS, PTB, etc.)
        if status['status'][1] == '0':  # sync_unspec
            j_status.append(CRIT)
            msg.append('no sync')
    else:
        msg.append('warming up')
    msg.append(
        'status: {}; stratum: {}; offset: {}; uptime: {}'.format(status['status'],
                                                                 status['stratum'],
                                                                 status['offset'],
                                                                 ntp_uptime))
    return Status(max(j_status)), ', '.join(msg)


@as_check(name=CHECK_NAME)
def juggler_check():
    try:
        ntpd_data = collect_ntp_data(('rv', 'as', 'sysstat'))
    except Exception as e:
        return WalleCompatEvent(Status.CRIT, 'Exception: {}'.format(str(e)))
    ntpd_associations = parse_as_output(ntpd_data['as'])
    ntpd_status.update(parse_rv_output(ntpd_data['rv']))
    ntpd_status.update(parse_sys_output(ntpd_data['sysstat']))
    return WalleCompatEvent(*make_decision(ntpd_status, ntpd_associations))
