"""
Checking that portod tasks not in porto container cgroups
"""
from juggler.bundles import as_check, Status
from infra.rtc.juggler.bundle.util import WalleCompatEvent
from os import listdir, path
from re import search

import porto


CHECK_NAME = 'portod_tasks'
PORTOCTL = '/usr/sbin/portoctl'


def read_and_close(path):
    with open(path, 'r') as f:
        return f.read()


def get_porto_pids():
    portod_pid = read_and_close('/run/portod.pid')
    portoloop_pid = read_and_close('/run/portoloop.pid')
    return [int(portod_pid), int(portoloop_pid)]


def get_porto_tasks(pids):
    porto_tids = []
    for pid in pids:
        proc = '/proc/{}/task'.format(pid)
        pids_tids = listdir(proc)
        porto_tids += pids_tids
    return porto_tids


def get_cgroups(task):
    raw_cgroups = read_and_close('/proc/{}/cgroup'.format(task)).split()
    porto_cgroups = [i.strip() for i in raw_cgroups]
    return porto_cgroups


def check_porto():
    c = porto.Connection(timeout=20)
    try:
        c.connect()
        state = c.GetData('/', 'state')
    except Exception as e:
        return 2, "Porto communication error: {}".format(e)
    if state != 'meta':
        return 2, "'/' container is not meta"
    return 0, "Ok"


@as_check(name=CHECK_NAME)
def juggler_check():
    if not path.exists(PORTOCTL):
        return WalleCompatEvent(Status.CRIT, "'{}' does not exist".format(PORTOCTL))

    status, msg = check_porto()
    if status:
        return WalleCompatEvent(Status.CRIT, msg)

    try:
        portod_pids = get_porto_pids()
        tasks = get_porto_tasks(portod_pids)

        for task in set(tasks):
            cgroups = get_cgroups(task)
            for cgroup in cgroups:
                # check that this cgroup is cgroups of the porto container
                if search('porto[%/]', cgroup):
                    return WalleCompatEvent(Status.CRIT, 'Incorrect cgroup for portod task {}: {}'.format(task, cgroup))
    except:
        pass

    return WalleCompatEvent(Status.OK, 'Ok')
