"""
Hotfixes via walle.

Reserved for future use.

"""

import time
import json
import os
import errno
import subprocess

from juggler.bundles import as_check, Status, Event

STATUS_MAP = {
    Status.OK: 'OK',
    Status.WARN: 'WARN',
    Status.CRIT: 'CRIT'
}
CHECK_NAME = 'walle_hotfix'


class PortoCondition(object):
    CGROUP_ROOT = '/sys/fs/cgroup/memory'
    AFFECTED_KERNELS = ['4.4.171-70.1', '4.4.114-50']

    def _is_host_affected(self):
        return os.uname()[2] in self.AFFECTED_KERNELS

    @staticmethod
    def _subdirs(path):
        try:
            for entry in os.scandir(path):
                if not entry.name.startswith('.') and entry.is_dir():
                    yield entry.path
        except EnvironmentError as e:
            if e.errno == errno.ENOENT:
                return
            else:
                raise

    def _find_relevant_cgroups(self):
        stack = [self.CGROUP_ROOT]
        while stack:
            root = stack.pop()
            for path in self._subdirs(root):
                stack.append(path)
                yield path

    def _find_underflow(self):
        memtotal = 0
        try:
            with open('/proc/meminfo', buffering=4096) as f:
                for l in f:
                    if "MemTotal" in l:
                        # It's always kB
                        memtotal = int(l.split()[1]) * 1024
                        break
        except EnvironmentError as e:
            return False, 'failed to read /proc/meminfo: {}'.format(e)
        if not memtotal:
            return False, 'failed to grep MemTotal from /proc/meminfo'

        for cgroup in self._find_relevant_cgroups():
            p = '{}/memory.anon.usage'.format(cgroup)
            try:
                with open(p) as stream:
                    buf = stream.read()
            except EnvironmentError as e:
                if e.errno == errno.ENOENT:
                    continue
                return False, 'failed to read {}: {}'.format(p, e)

            try:
                anonusage = int(buf)
            except Exception as e:
                return False, 'failed to parse {} content: {}'.format(p, e)

            if anonusage > memtotal:  # or anonusage & 0x1000000000000000L (aka > 1 ** 63)
                return True, '{} has abnormal memory usage'.format(cgroup)

        return False, None

    def need_hotfix(self):
        if not self._is_host_affected():
            return False, None
        return self._find_underflow()


class NetnsCondition(object):
    def need_hotfix(self):
        try:
            res = subprocess.Popen(['sudo', '-n', 'unshare', '-n', 'true'],
                                   stdin=subprocess.DEVNULL,
                                   stdout=subprocess.DEVNULL,
                                   stderr=subprocess.STDOUT).wait(60)
            if res:
                return True, "failed to check netns creation"
        except subprocess.TimeoutExpired:
            return True, "failed to create netns in 60 seconds, probably KERNEL-151"
        return False, None


CONDITIONS = [
    PortoCondition,
    NetnsCondition
]


@as_check(name=CHECK_NAME)
def juggler_check():
    timestamp = time.time()

    status = Status.OK
    reason = ''
    for cls in CONDITIONS:
        condition = cls()
        needed, hotfix_reason = condition.need_hotfix()
        if needed:
            status = Status.CRIT
            reason = hotfix_reason
            break
        elif needed is None:
            status = Status.WARN
            reason = hotfix_reason

    description = json.dumps({
        'status': STATUS_MAP[status],
        'timestamp': timestamp,
        'reason': reason or 'passed'
    })
    return Event(status, description)
