#!/usr/bin/env python
# Provides: bad_raid_check
# -*- coding: utf-8 -*-

import os
import re

CHECK_NAME = 'bad_raid_check'


def die(check_name, status, msg):
    print("PASSIVE-CHECK:%s;%s;%s" % (check_name, status, msg))
    raise SystemExit(0)


def readline_from_file(f_name, max_size=1024):
    try:
        with open(f_name) as f:
            return f.readline(max_size).strip()
    except Exception as e:
        die(CHECK_NAME, 1, 'failed to read "{}": {}'.format(f_name, e))


def list_disks():
    disks = {}
    for name in os.listdir('/sys/block'):
        if not name.startswith('sd'):
            continue
        check_ssd = readline_from_file('/sys/block/{}/queue/rotational'.format(name))
        if check_ssd.startswith('0'):
            disks.update({name: {'ssd': True}})
        else:
            disks.update({name: {'ssd': False}})
    return disks


def main():
    # Check "SSD-on-RAID0" only for /^web-.*/ lui-profiles
    check_raid0 = True
    lui_conf = "/etc/lui.conf"
    if os.path.isfile(lui_conf):
        for line in open(lui_conf, 'r'):
            m = re.match("^# Name:\s+(.+)\s+Revision:", line)
            if m:
                if not re.match("^web-", m.group(1)):
                    check_raid0 = False
                break

    disks = list_disks()
    with open('/proc/mdstat') as mdstat:
        for line in mdstat:
            if not line.startswith('md'):
                continue
            line_arr = line.split()
            md_name = line_arr[0]
            md_type = line_arr[3]
            md_partisipant = []
            md_disks_type = []
            for disk in line_arr[4:]:
                md_partisipant.append(disk[:4])
                if disk[:3] not in disks:
                    continue  # skip failed disks
                if disks[disk[:3]]['ssd'] not in md_disks_type:
                    md_disks_type.append(disks[disk[:3]]['ssd'])

            if check_raid0 and md_type != 'raid0':
                if True in md_disks_type:
                    msg = 'raid {} have {} type and made of ssd('.format(md_name, md_type)
                    for ds in md_partisipant:
                        msg += '{} - {};'.format(ds, 'SSD' if disks[ds[:3]]['ssd'] else 'Disk')
                    msg += ')'
                    die(CHECK_NAME, 2, msg)

            if len(md_disks_type) == 2:
                msg = 'raid {} have different type of disks('.format(md_name)
                for ds in md_partisipant:
                    msg += '{} - {};'.format(ds, 'SSD' if disks[ds[:3]]['ssd'] else 'Disk')
                msg += ')'
                die(CHECK_NAME, 2, msg)

    die(CHECK_NAME, 0, 'Ok')


if __name__ == '__main__':
    main()
