#!/skynet/python/bin/python2
# -*- coding: utf-8 -*-
# Provides: check_iss_agent

import urllib2
import sys
import subprocess
import json
import porto
import time


CHECK_NAME = "check_iss_agent"

ISS_AGENT_PORT = 25536
ISS_AGENT_CONTAINER = 'iss-agent'
ISS_AGENT_CONTAINER_EXPECTED_STATE = 'running'

ISS_SYNC_THRESHOLD_SEC = 300
ISS_REQUEST_TIMEOUT_SEC = 15

PORTO_REQUEST_TIMEOUT_SEC = 5
CHECK_CONTAINER_RETRIES = 3
CHECK_CONTAINER_DELAY_SEC = 3

ISS_MASTER_CONNECTION_THRESHOLD_MSEC = 30000


def die(status, message):
    timestamp = time.time()
    result = json.dumps({"status": status, "timestamp": int(timestamp), "reason": message})
    print('PASSIVE-CHECK:{};{};{}'.format(CHECK_NAME, status, result))
    sys.exit(0)


def call_iss_agent_rest(path):
    url = 'http://localhost:{}{}'.format(ISS_AGENT_PORT, path)
    try:
        return urllib2.urlopen(url, timeout=ISS_REQUEST_TIMEOUT_SEC)
    except Exception as e:
        die(2, "Get URL '{}' failed: {}".format(url, e))


def get_json_from_agent_rest(path):
    response = call_iss_agent_rest(path)
    try:
        return json.loads(response.read())
    except Exception as e:
        die(2, "Could not parse JSON response from '{}' path: {}".format(path, e))


def check_package():
    try:
        res = subprocess.Popen(
            ["dpkg-query", "--show", "--showformat", "'${Version}'", "iss-agent"],
            shell=False,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE
        )
        stderr = res.stderr.read()
        if stderr.startswith('No packages found'):
            die(2, 'iss-agent package was not found')

    except OSError:
        die(2, 'failed to check iss-agent package')


def check_in_porto(func):
    p_api = porto.Connection(timeout=PORTO_REQUEST_TIMEOUT_SEC)
    try:
        p_api.connect()
    except porto.exceptions.SocketTimeout:
        die(2, 'Failed to connect Porto. Perhaps it is missing on the host')

    err_code = 0
    err_msg = ''

    for _ in range(CHECK_CONTAINER_RETRIES):
        try:
            err_code, err_msg, need_retry = func(p_api)
            if not need_retry:
                break

        except porto.exceptions.ContainerDoesNotExist:
            err_code = 2
            err_msg = "'{}' container not found".format(ISS_AGENT_CONTAINER)

        except Exception as e:
            err_code = 2
            err_msg = "Failed to get '{}' container state: {}".format(ISS_AGENT_CONTAINER, e)

        time.sleep(CHECK_CONTAINER_DELAY_SEC)

    if err_code:
        die(err_code, err_msg)


def _check_container(p_api):
    state = p_api.GetData(ISS_AGENT_CONTAINER, 'state')
    mem_limit = float(p_api.GetData(ISS_AGENT_CONTAINER, 'memory_limit'))

    err_code = 0
    err_msg = ''
    need_retry = False

    if state != ISS_AGENT_CONTAINER_EXPECTED_STATE:
        err_code = 2
        err_msg = "{} container state: '{}' != '{}'".format(
            ISS_AGENT_CONTAINER, state, ISS_AGENT_CONTAINER_EXPECTED_STATE)
        need_retry = True
    elif mem_limit == 0:
        err_code = 1
        err_msg = 'memory limit set to zero'

    return err_code, err_msg, need_retry


def _check_jit_enabled(p_api):
    stdout = p_api.GetData(ISS_AGENT_CONTAINER, 'stdout')
    stderr = p_api.GetData(ISS_AGENT_CONTAINER, 'stderr')

    err_code = 0
    err_msg = ''
    need_retry = False

    if 'Compiler has been disabled' in stderr or 'compilation: disabled' in stdout:
        err_code = 2
        err_msg = "JIT compilation disabled"

    return err_code, err_msg, need_retry


def check_sync_timing(stats):
    sync_duration = stats['agent.sync_duration_axxx']
    if sync_duration > ISS_SYNC_THRESHOLD_SEC*1000:
        die(2, "sync duration {} more than {} seconds".format(int(sync_duration/1000), ISS_SYNC_THRESHOLD_SEC))


def check_master_links(stats):
    for source in ['yp', 'iss']:
        alive = stats.get('agent.provider.{}.alive_ahhh'.format(source), 1)
        dead_time = stats.get('agent.provider.{}.no_connection_ms_ahhh'.format(source), 0)
        if not alive and dead_time > ISS_MASTER_CONNECTION_THRESHOLD_MSEC:
            die(2, "no connection to {} master more than {} seconds".format(source, int(dead_time/1000)))


def get_stats():
    stats = get_json_from_agent_rest('/unistat')
    stats_dict = {v[0]: v[1] for v in stats}
    return stats_dict


def main():
    try:
        check_package()
        check_in_porto(_check_container)
        check_in_porto(_check_jit_enabled)

        stats = get_stats()
        check_sync_timing(stats)
        check_master_links(stats)

        die(0, "OK")
    except SystemExit:
        # ignore system exit exception
        pass
    except BaseException as ex:
        # do not throw exceptions if something goes wrong, juggler expects metadata
        die(2, "Something went wrong: " + str(ex))


if __name__ == '__main__':
    main()
