#!/usr/bin/env python
# -*- coding: utf-8 -*-
# Provides: check_network_irq

import os
import collections

CHECK_NAME = 'check_network_irq'

THRESHOLD_WARN = 0.7
THRESHOLD_CRIT = 0.3


class IrqNotFound(Exception):
    pass


class LinuxSys(object):
    SYS_CLASS_NET = "/sys/class/net"
    SYS_DEVICES_VIRTUAL_NET = "/sys/devices/virtual/net/"
    PROC_IRQ = '/proc/irq'


class IRQ(LinuxSys):
    def __init__(self, irq):
        """
        :param irq: string\int
        """
        self.irq = irq
        self.path_irq = '{}/{}'.format(self.PROC_IRQ, self.irq)
        if not os.path.exists(self.path_irq):
            raise IrqNotFound('Not found IRQ {} '.format(self.path_irq))

    def __get_spurious(self):
        """
        :return: dict
        """
        metrics = {}
        with open('{}/spurious'.format(self.path_irq), 'r') as f:
            for line in f:
                fields = line.strip().split()
                metrics[fields[0]] = ' '.join(fields[1:])
        return metrics

    def count(self):
        """
        Count interrupt
        :return: int
        """
        return int(self.__get_spurious()['count'])

    def smp_affinity(self):
        """
        :return: string
        """
        with open('{}/smp_affinity'.format(self.path_irq), 'r') as f:
            content = f.read().rstrip()
        return content

    def smp_affinity_list(self):
        """
        :return: string
        """
        with open('{}/smp_affinity_list'.format(self.path_irq), 'r') as f:
            content = f.read().rstrip()
        return content

    def __str__(self):
        """
        Return IRQ
        """
        return self.irq


class NetDev(LinuxSys):

    def __init__(self, name):
        """
        :param name: Network name "eth", "vlan" ... etc
        """
        self.name = name
        self.class_net_dev = None
        self.device_path_dev = None
        self.class_net_dev = '{}/{}'.format(self.SYS_CLASS_NET, self.name)

        if not os.path.islink(self.class_net_dev):
            raise Exception('Not found dev {}'.format(self.name))

        self.device_path_dev = os.path.realpath(self.class_net_dev)

    def is_virtual(self):
        """
        :return: Bool
        """
        if self.SYS_DEVICES_VIRTUAL_NET in self.device_path_dev:
            return True
        return False

    def is_up(self):
        """
        Return state
        :return:
        """
        with open('{}/operstate'.format(self.device_path_dev), 'r') as f:
            content = f.read()
        if content.rstrip().lower() == 'up':
            return True
        return False

    def list_irqs(self):
        """
        Return list object IRQ
        :return:
        """
        if self.is_virtual() or not self.is_up():
            return None
        msi_irqs_path = "{}/device/msi_irqs".format(self.class_net_dev)
        list_irq = []
        for f in os.listdir(msi_irqs_path):
            try:
                list_irq.append(IRQ(irq=f))
            except IrqNotFound:
                continue
        return list_irq

    def __repr__(self):
        """
        :return:
        """
        str = "Name : {}\n".format(self.name)
        str += "Device path : {}\n".format(self.device_path_dev)
        str += "Class path (sym) : {}\n".format(self.class_net_dev)
        str += "Virtual device : {}\n".format(self.is_virtual())
        str += "Operstate UP: {}\n".format(self.is_up())
        # str += "List IRQ : {}".format(', '.format(self.list_irqs()))
        return str


def check():
    for f in os.listdir(LinuxSys.SYS_CLASS_NET):
        nd = NetDev(f)
        if nd.is_virtual() or not nd.is_up():
            continue
        smp_aff_l = []
        for irq in nd.list_irqs():
            # print "{} {} smp_aff {}".format(irq.irq, irq.count(), irq.smp_affinity_list())
            smp_aff_l.append(irq.smp_affinity_list())

        c = collections.Counter(smp_aff_l)
        len_c = len(c.keys())
        if len_c == 1:
            # 0.0% interrupt dist
            len_c = 0

        pr = float(len_c) / len(smp_aff_l)
        if pr <= THRESHOLD_CRIT:
            return 2, "Interface {} interrupt distribution {:.2f}%".format(nd.name, pr * 100)
        elif pr <= THRESHOLD_WARN:
            return 1, "Interface {} interrupt distribution {:.2f}%".format(nd.name, pr * 100)
    return 0, "OK"


def die(check_name, status, message):
    print 'PASSIVE-CHECK:%s;%s;%s' % (check_name, status, message)
    raise SystemExit(0)


if __name__ == '__main__':
    status, message = check()
    die(check_name=CHECK_NAME, status=status, message=message)
