#!/bin/sh
#
# Provides: cpu_state
#
# $Id$
# $HeadURL$
#
# https://st.yandex-team.ru/SEPE-8576

me=${0##*/}
me=${me%.*}

HOME="/home/monitor"

die () {
    if [ "${unix_style}" = "YES" ]; then
        echo >&2 "$2"
        exit $1
    else
        echo "PASSIVE-CHECK:$me;$1;$2"
        exit 0
    fi
}

[ `uname` = Linux ] || die 1 "OS not supported"
export LANG=C
export LC_ALL=C

while getopts c:hu opt; do
    case ${opt} in
    c)
        conf_file="${OPTARG}"
        [ -f "${conf_file}" -a -r "${conf_file}" ] || \
            die 2 "Defined conf file not readable or not exists '${conf_file}'" # user must know
        ;;
    u)
        unix_style="YES"
        ;;
    h|*)
        printf "${me} -- Check CPU trottling/cpufreq states, https://st.yandex-team.ru/SEPE-8576\n\n"
        printf "Usage: ${me} [-u] [-c <conf_file>]\nOptions: \n"
        printf "\t-c\tPath to configuration file\n"
        printf "\t-u\tUnix style STDERR and exit codes (NAGIOS/JUGGLER style by default)\n"
        exit 1
        ;;
    esac
done

: ${conf_file="${HOME}/agents/etc/${me}.conf"}
if [ -f "${conf_file}" -a -r "${conf_file}" ]; then
    # source (.) in Linux sh fails on paths w/o slashes. thats why eval here
    eval "`cat "${conf_file}"`" || die 2 "Failed to load conf file '${conf_file}'"
fi

# defaults
: ${idle_thresh="15"} # percents
: ${permitted_scaling_governors="performance"} # 'ondemand' will cause crit, for example
: ${keep_last_states="2"}

[ -w "/dev/shm" ] && states_path="/dev/shm" 
: ${states_path="${HOME}/agents/tmp"} # empty -- disable

grab_files() {
    local _f_body _file _glob
    [ -n "${1}" ] && { _glob="${1}"; shift; } || return 1
    for _file in ${_glob}; do
        [ -r "${_file}" ] || continue; # may not exists
        _f_body="`cat "${_file}"`"
        [ ${?} -gt 0 ] && return 1
        echo "${_file}:${_f_body}"
    done
}

rotate_file() {
    local _file _state
    [ -n "${1}" ] && { _file="${1}"; shift; } || return 1
    for _state in `seq ${keep_last_states} -1 0`; do
        [ -f "${_file}.$((${_state} - 1))" ] || continue
        mv -f "${_file}.$((${_state} - 1))" "${_file}.${_state}" || return 1
    done
}

check_counters() {
    local _glob _name _descr
    [ -n "${1}" ] && { _glob="${1}"; shift; } || return 1
    [ -n "${1}" ] && { _name="${1}"; shift; } || return 1
    [ -n "${1}" ] && { _descr="${1}"; shift; } || return 1
    _counters="`grab_files "${_glob}"`"
    [ ${?} -gt 0 ] && { crit_msg="${crit_msg}Failed to read files '${_glob}'; "; return 1; }
    if [ -n "${states_path}" ]; then
        rotate_file "${states_path}/${me}.${_name}" || \
            { crit_msg="${crit_msg}Failed to rotate states for '${_name}' counts; "; return 1; }
        echo "${_counters}" > "${states_path}/${me}.${_name}.0" || \
            { crit_msg="${crit_msg}Failed to write state for '${_name}' counts; "; return 1; }
        if [ -f "${states_path}/${me}.${_name}.${keep_last_states}" -a -f "${states_path}/${me}.${_name}.0" ]; then
            _counters="`diff 2>/dev/null "${states_path}/${me}.${_name}.${keep_last_states}" "${states_path}/${me}.${_name}.0"`"
            _counters="`echo "${_counters}" | grep "^> " | sed "s/^> //"`"
        else
            _counters=""
        fi
        _counters=`echo "${_counters}" | grep cpu | wc -l`;
    else
        _counters=`echo "${_counters}" | awk -F '/|:' '$9 != 0 {print $6}' | wc -l`;
    fi
    [ "${_counters}" -lt 1 ] && return 0
    crit_msg="${crit_msg}${_descr} detected on ${_counters} CPUs; "
}

check_frequency() {
    local _core_freq _cpu_idle _low_freq _low_freq_cores _min_freq _min_freq_cores _total_cores
    _cpu_idle=`sar -u 1 1 | awk '$1 == "Average:" {print int($NF)}'`
    info_msg="${info_msg}cpu_idle:${_cpu_idle}; "

    _min_freq=`cat /sys/devices/system/cpu/cpu0/cpufreq/cpuinfo_min_freq 2>/dev/null`
    [ -n "${_min_freq}" ] || return 0 # don't fail on old CPUs

    _low_freq=$((${_min_freq} * 2)) # cpuinfo_min_freq at least twice lower than nominal frequency.. usually =)
    _low_freq=$((${_low_freq} - 150000)) # substract 150MHz to fit E5-2660 v4 and reduce false-positives
    _min_freq=$((${_min_freq} + 1000)) # core sensors may show a tiny bit higher freq - neutralize such noise

    _total_cores=0
    _low_freq_cores=0
    _min_freq_cores=0

    for _core_freq in `awk -F : '$1 ~ "^cpu MHz" {print int($2*1000)}' /proc/cpuinfo`; do
        [ ${_core_freq} -lt ${_low_freq} ] && _low_freq_cores=$((${_low_freq_cores} + 1))
        [ ${_core_freq} -le ${_min_freq} ] && _min_freq_cores=$((${_min_freq_cores} + 1))
        _total_cores=$((${_total_cores} + 1))
    done

    [ ${_cpu_idle} -gt ${idle_thresh} ] || \
        warn_msg="${warn_msg}CPU idle < ${idle_thresh}%; "

    # at least one core should be loaded and had freq higher than minimum (I'm running or not!? =)
    [ ${_min_freq_cores} -eq ${_total_cores} ] && \
        crit_msg="${crit_msg}All CPUs at their minimal frequency; " && \
        return # low freq check below meaningless when this check triggered

    # alert when cpu idle less than 50% (even with governor 'performance' frequency may be dropped on low load)
    [ ${_low_freq_cores} -gt 0 -a ${_cpu_idle} -lt 50 ] && \
        crit_msg="${crit_msg}Low frequency on ${_low_freq_cores} CPUs; "
}

check_scaling_governor() {
    local _gov _governs
    [ -n "${permitted_scaling_governors}" ] || return 0
    _governs="`grab_files "/sys/devices/system/cpu/cpu[0-9]*/cpufreq/scaling_governor"`"
    [ ${?} -gt 0 ] && { crit_msg="${crit_msg}Failed to read scaling_governor files"; return 1; }
    for _gov in ${permitted_scaling_governors}; do
        _governs="`echo "${_governs}" | grep -wv ${_gov}`" # remove all permitted
    done
    [ -z "${_governs}" ] && return 0
    _governs="`echo "${_governs}" | awk -F : '{print $2}' | sort | uniq -c | awk '{print $2" on "$1" CPUs"}' | tr "\n" , | sed \"s/,$//\"`"
    crit_msg="${crit_msg}Wrong scale governors detected: ${_governs}; "
}

check_temperature() {
    local _count _core _core_ids _core_t_avg _core_t_crit _core_t_curr _reached_near_crit _reached_crit
    _core_t_avg=""; _reached_crit=""; _reached_near_crit=""
    _core_ids=`get_cpu_temps`
    [ $? -ne 0 -o -z "${_core_ids}" ] && { warn_msg="${warn_msg}Failed to get temperature data via hwmon; "; return 1; }

    for _core in ${_core_ids}; do
        set -- `echo ${_core} | sed "s/:/ /"`
        _core_t_curr=$1; _core_t_crit=$2
        _core_t_avg=$(( $_core_t_avg + $_core_t_curr ))
        if [ ${_core_t_curr} -ge $((${_core_t_crit} - 3)) ]; then
            if [ "${_core_t_curr}" -ge "${_core_t_crit}" ]; then
                _reached_crit="${_reached_crit} ${_core_t_curr}:${_core_t_crit}"
            else
                _reached_near_crit="${_reached_near_crit} ${_core_t_curr}:${_core_t_crit}"
            fi
        fi
    done

    if [ -n "${_core_ids}" ]; then
        _core_t_avg=$(( ${_core_t_avg} / `echo ${_core_ids} | wc -w` ))
        info_msg="${info_msg}cpu_temp:${_core_t_avg}; "
    else
        info_msg="${info_msg}cpu_temp:N/A; "
    fi

    if [ -n "${_reached_crit}" ]; then
        _count=`echo ${_reached_crit} | wc -w`
        crit_msg="${crit_msg}At least ${_count} CPUs reached crit temperature (curr:crit): ${_reached_crit}; "
    fi
    if [ -n "${_reached_near_crit}" ]; then
        _count=`echo ${_reached_near_crit} | wc -w`
        crit_msg="${crit_msg}At least ${_count} CPUs near crit temperature (curr:crit): ${_reached_near_crit}; "
    fi
}

check_turbo_boost() {
    [ -f "/sys/devices/system/cpu/intel_pstate/no_turbo" ] || return 0 # not supported at all
    [ "`cat /sys/devices/system/cpu/intel_pstate/no_turbo 2>/dev/null`" = 0 ] || \
        crit_msg="${crit_msg}TurboBoost disabled; "
}

# returns list of elemnts, element is "$t_curr:$t_crit"
get_cpu_temps() {
    local _core _core_ids _hwmon _t_curr _t_crit _t_path
    for _hwmon in /sys/class/hwmon/hwmon*; do
        [ -L "${_hwmon}/device/driver" ] || continue

        # searching sensors files
        for _t_path in . device; do # diffenent paths for kernels: 3.10 -- device/, 3.18 -- ./
            _core_ids=`ls -d ${_hwmon}/${_t_path}/temp*_input 2>/dev/null| awk -F / '{print $NF}' | tr -d '[a-zA-Z_]' | sort` 2>/dev/null
            [ -n "${_core_ids}" ] && break # found
        done
        [ -n "${_core_ids}" ] || continue

        # verify drivers (hwmon supports not only CPU sensors)
        case $(basename `readlink ${_hwmon}/device/driver`) in
        coretemp)
            [ -n "`ls -d ${_hwmon}/${_t_path}/temp*_max 2>/dev/null`" ] || continue ;;
        k*temp) # AMD, k10temp
            [ -n "`ls -d ${_hwmon}/${_t_path}/temp*_crit_hyst 2>/dev/null`" ] || continue ;;
        *) # unsupported type, skip
            continue ;;
        esac

        # report
        for _core in ${_core_ids}; do
            _t_curr=`cat ${_hwmon}/${_t_path}/temp${_core}_input 2>/dev/null`
            [ -f "${_hwmon}/${_t_path}/temp${_core}_crit" ] && \
                _t_crit=`cat ${_hwmon}/${_t_path}/temp${_core}_crit 2>/dev/null` || \
                _t_crit=`cat ${_hwmon}/${_t_path}/temp${_core}_max 2>/dev/null` # assume max == crit if crit no avail
            if [ -n "${_t_curr}" -a -n "${_t_crit}" ]; then
                echo -n "$((${_t_curr} / 1000)):$((${_t_crit} / 1000)) " # from milicelsius
            else
                warn_msg="${warn_msg}Failed to get temperature data from '${_hwmon}/${_t_path}/temp${_core}'; "
                return 1
            fi
        done
    done
}

crit_msg=""
info_msg=""
warn_msg=""

[ -n "${states_path}" -a -d "${states_path}" -a -w "${states_path}" ] || \
{ warn_msg="${warn_msg}Path for states is not writable. States disabled. "; states_path=""; }

check_scaling_governor

check_counters "/sys/devices/system/cpu/cpu[0-9]*/thermal_throttle/core_throttle_count" \
    "core_throttle_count" "Core throttling"

check_counters "/sys/devices/system/cpu/cpu[0-9]*/thermal_throttle/core_power_limit_count" \
    "core_power_limit" "Power limits"

check_temperature

check_frequency

check_turbo_boost

if [ -n "${crit_msg}" ]; then
    [ -n "${warn_msg}" ] && crit_msg="CRITS:: ${crit_msg}WARNS:: ${warn_msg}"
    die 2 "${crit_msg}INFO:: ${info_msg}"
fi
[ -n "${warn_msg}" ] && die 1 "${warn_msg}INFO:: ${info_msg}"
die 0 "Ok; INFO:: ${info_msg}"
