#!/usr/bin/env python
# Provides: fastbone
# -*- coding: utf-8 -*-
#
# This script pings fastbone hosts returns 1 if any of hosts are
# defined but not reachable.
# pylint: disable=invalid-name,line-too-long,missing-docstring
#
import os
import re
import subprocess
import urllib2
import json
from xml.etree import ElementTree


CHECK_NAME = 'fastbone'
RC_CONF_LOCAL = '/etc/rc.conf.local'
YA_SUBR = '/etc/ya.subr'
YA_SUBR_JSON = '/usr/lib/yandex-netconfig/patches/yasubr.json'
# Use netconfig cache file until RUNTIMECLOUD-6652 is resolved
NETWORKS_MAP_FILE = '/var/cache/network/ya-netconfig/networks.json'
DEFAULT_CHECK_HOST = 'fb-jnob-juggler.yandex.ru'
RACKTABLES_URL = "https://racktables.yandex.net/export/get-vlanconfig-by-port.php?switch=%s&port=%s"
LLDP_INTERFACES_RE = 'eth'
CHECK_HOST_MAP = {
    "vlan865": None,  # ignore vlan's without hosts to ping
    "vlan866": None,
    "vlan788": None,
}


def die(check_name, status, msg):
    print("PASSIVE-CHECK:%s;%s;%s" % (check_name, status, msg))
    raise SystemExit(0)


def parse_lldpctl_output(output):
    root = ElementTree.fromstring(output.strip())
    if root.tag != "lldp":
        raise ValueError("Invalid root element: {}.", root.tag)

    ifaces = set()
    switches = set()

    for iface_element in root.iter("interface"):
        iface = iface_element.attrib["name"]

        if not re.match(LLDP_INTERFACES_RE, iface):
            continue

        ifaces.add(iface)

        chassis_elements = iface_element.findall("chassis")
        chassis_names = chassis_elements[0].findall("name")

        switch = chassis_names[0].text
        if not switch:
            switch = ""

        port_elements = iface_element.findall("port")
        port_id_elements = port_elements[0].findall("id")
        port_id_element = port_id_elements[0]

        port = port_id_element.text
        if not port:
            port = ""

    vlans = set()
    if iface_element.findall("vlan"):
        for vl in iface_element.iter("vlan"):
            vlan = vl.attrib['vlan-id']
            vlans.add("vlan" + vlan)
            switches.add((switch, port, vlan))
    else:
        switches.add((switch, port, ""))

    return [{"switch": switch, "port": port, "vlans": vlans} for switch, port, vlan in switches]


def get_racktables_info(switch, port):
    result = urllib2.urlopen(RACKTABLES_URL % (switch, port))
    vlans_str = result.read().split('\n')[0]

    vlans_str = vlans_str.replace('T', '')
    vlans_str = vlans_str.replace('A', '')
    vlans_str = vlans_str.replace('+', ' ')
    vlans_str = vlans_str.replace('-', ' ')
    vlans_str = vlans_str.replace(',', ' ')

    vlans = vlans_str.split()
    vlans = map(lambda v: 'vlan' + str(v), vlans)
    return vlans


def run_command(cmd):
    try:
        process = subprocess.Popen(cmd,
                                   stdin=subprocess.PIPE,
                                   stdout=subprocess.PIPE,
                                   stderr=subprocess.PIPE,
                                   close_fds=True)
        stdout, stderr = process.communicate()
    except Exception:
        return None
    stdout = stdout.decode("utf-8")
    return stdout


def get_switch_info():
    try:
        output = run_command(["sudo", "lldpctl", "-f", "xml"])
        return parse_lldpctl_output(output)
    except Exception:
        return None


def fb_interfaces():
    interfaces = []
    all_interfaces = os.listdir('/sys/class/net/')
    use_groups = False
    with open('/etc/iproute2/group') as f:
        out = f.read()
    for line in out.split('\n'):
        line = line.strip()
        if line == '':
            continue
        if line.startswith('#'):
            continue
        if line[0] == '1' or line[1] == '2':
            use_groups = True

    for iface in all_interfaces:
        with open('/sys/class/net/' + iface + '/operstate') as f:
            status = f.read()
            status = status.strip()
        with open('/sys/class/net/' + iface + '/netdev_group') as f:
            group = f.read()
            group = group.strip()

        if use_groups:
            if group == '2' and status == 'up':
                interfaces.append(iface)
        else:
            if iface.startswith('vlan') and status == 'up':
                interfaces.append(iface)
    if not interfaces:
        die(CHECK_NAME, 2, "Can't detect fb interface")
    interfaces.sort()
    return interfaces


def real_interfaces():
    interfaces = []
    all_interfaces = os.listdir('/sys/class/net/')
    use_groups = False
    with open('/etc/iproute2/group') as f:
        out = f.read()
    for line in out.split('\n'):
        line = line.strip()
        if line == '':
            continue
        if line.startswith('#'):
            continue
        if line[0] == '1' or line[1] == '2':
            use_groups = True

    for iface in all_interfaces:
        with open('/sys/class/net/' + iface + '/operstate') as f:
            status = f.read()
            status = status.strip()
        with open('/sys/class/net/' + iface + '/netdev_group') as f:
            group = f.read()
            group = group.strip()
        if use_groups:
            if group == '1' and status == 'up':
                interfaces.append(iface)
        else:
            if iface.startswith('eth') and status == 'up':
                interfaces.append(iface)
    if not interfaces:
        die(CHECK_NAME, 2, "Can't detect real interface")
    interfaces.sort()
    return interfaces


def get_lldp_info():
    _cmd = 'sudo /usr/sbin/lldpctl -f xml'
    links = subprocess.Popen(
        _cmd.split(),
        stdout=subprocess.PIPE,
        stderr=subprocess.PIPE
    )
    out, _ = links.communicate()


def check_vlan_on_port(interface):
    switch_info = get_switch_info()
    if switch_info is not None:
        info = switch_info[0]
        vlans = get_racktables_info(info['switch'], info['port'])
        if interface not in vlans:
            die(CHECK_NAME, 2, "Can't find vlan: %s on switch/port (%s / %s)" % (interface, info['switch'], info['port']))
        return 0, "Ok"
    else:
        return 1, "Can't get vlan info"


def ping(interface, size, host):
    ping_cmd = 'ping6 -q -n -I %s -s %s -c 4 %s' % (interface, size, host)
    ping_proc = subprocess.Popen(
        ping_cmd.split(),
        stdout=subprocess.PIPE,
        stderr=subprocess.PIPE
    )
    out_big_ping, error = ping_proc.communicate()
    if ping_proc.returncode != 0:
        size = 1450
        ping_cmd = 'ping6 -q -n -I %s -s %s -c 4 %s' % (interface, size, host)
        ping_proc = subprocess.Popen(
            ping_cmd.split(),
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE
        )
        out, error = ping_proc.communicate()
        if ping_proc.returncode == 0:
            return 2, "Can't ping %s with big packets: %s" % (host, out_big_ping)
        else:
            # Check vlan on switch/port
            (code, msg) = check_vlan_on_port(interface)
            ret_msg = "Can't ping %s: %s" % (host, (error or out).strip())
            if code == 1:
                ret_msg = msg + "; " + ret_msg
            return 2, ret_msg
    return 0, "Ok"


def ping_all(gates=None):
    if gates is None:
        gates = {}
    status = 0
    for fb_int in fb_interfaces():
        host = CHECK_HOST_MAP.get(fb_int, DEFAULT_CHECK_HOST)
        if host is None:
            continue
        size = 8860
        (status, msg) = ping(fb_int, size, host)
        if status != 0:
            break
    if status == 0:
        die(CHECK_NAME, 0, "Ok")
    # see JUGGLER-1207 why we don't ping gateways
    else:
        for (gate, sets) in gates.items():
            size = sets['mtu'] - 50
            fb_int = sets['fb_int']
            if gate == 'first':
                gate = 'fe80::1'
            (status_g, msg_g) = ping(fb_int, size, gate)
            if status_g != 0:
                die(CHECK_NAME, status_g, msg_g)
        die(CHECK_NAME, status, msg)


def get_ipv6_addr_from_interface(interface):
    cmd = 'ip -6 addr show scope global %s' % interface
    show_global = subprocess.Popen(
        cmd.split(),
        stdout=subprocess.PIPE,
        stderr=subprocess.PIPE
    )
    output, err = show_global.communicate()
    if show_global.returncode == 0 and output:
        inet6str = output.splitlines()[1]
        ipv6str = inet6str.split()[1].split('/')[0]
    else:
        return None
    if ipv6str:
        return ipv6str


def get_ipv4_addr_from_interface(interface):
    cmd = 'ip -4 addr show scope global %s' % interface
    show_global = subprocess.Popen(
        cmd.split(),
        stdout=subprocess.PIPE,
        stderr=subprocess.PIPE
    )
    output, err = show_global.communicate()
    if show_global.returncode == 0 and output:
        inet4str = output.splitlines()[1]
        ipv4str = inet4str.split()[1].split('/')[0]
    else:
        return None
    if ipv4str:
        return ipv4str


def check_networks_json():
    try:
        from ipaddr import IPAddress, IPNetwork
        with open(NETWORKS_MAP_FILE) as f:
            networks_info = json.load(f)
    except ImportError:
        die(CHECK_NAME, 0, 'No ipaddr module, force OK: RUNTIMECLOUD-7912')
    except Exception as e:
        die(CHECK_NAME, 2, "Failed to read networks.json (%s)" % e)

    gates = {}
    interfaces = real_interfaces()
    for interface in interfaces:
        ip = get_ipv6_addr_from_interface(interface)
        if ip is None:
            die(CHECK_NAME, 2, "Can't find ip on {}".format(interface))
        ip = unicode(ip)
        for network in networks_info['data']:
            ipnet = IPNetwork(network)
            if IPAddress(ip) not in ipnet:
                continue
            if 'fastbone_vlans' not in networks_info['data'][network]:
                continue
            for fb_vlan, fb_info in networks_info['data'][network]['fastbone_vlans'].items():
                if 'routes' not in fb_info:
                    continue
                fb_route_groups = fb_info['routes']
                for fb_net, fb_route_info in networks_info['route_groups'][fb_route_groups].items():
                    gate = fb_route_info.get('gateway')
                    mtu = fb_route_info.get('mtu', None)
                    vlan = fb_vlan
                    if gate and gate not in gates:
                        gates.update({gate.decode('utf-8'): {'mtu': mtu, 'fb_int': 'vlan' + vlan}})
    ping_all(gates=gates)


def check_ya_subr_json():
    from ipaddr import IPAddress, IPNetwork
    with open(NETWORKS_MAP_FILE) as f:
        networks_info = json.load(f)
    with open(YA_SUBR_JSON) as f:
        yasubr_networks_info = json.load(f)

    for key in ['data', 'route_groups']:
        for info in yasubr_networks_info[key]:
            networks_info[key][info] = yasubr_networks_info[key][info]

    gates = {}
    interfaces = real_interfaces()
    for interface in interfaces:
        ip = get_ipv4_addr_from_interface(interface)
        if ip is None:
            die(CHECK_NAME, 2, "Can't find ip on {}".format(interface))
        ip = unicode(ip)
        for network in networks_info['data']:
            ipnet = IPNetwork(network)
            if IPAddress(ip) not in ipnet:
                continue
            for fb_vlan, fb_info in networks_info['data'][network]['fastbone_vlans'].items():
                if 'routes' not in fb_info:
                    continue
                fb_route_groups = fb_info['routes']
                for fb_net, fb_route_info in networks_info['route_groups'][fb_route_groups].items():
                    gate = fb_route_info.get('gateway')
                    mtu = fb_route_info.get('mtu', None)
                    vlan = fb_vlan
                    if gate and gate not in gates:
                        gates.update({gate.decode('utf-8'): {'mtu': mtu, 'fb_int': 'vlan' + vlan}})
    ping_all(gates=gates)


def get_ya_subr_settings():
    cmd = (
            '. /etc/ya.subr && . /etc/rc.conf.local && ' +
            'eval $(ya_network_info $ipaddr || ' +
            'echo exit 1) && printf "${ya_FB_ipv6_VLAN};${ya_FB_ipv6_defrouter}"'
    )
    proc = subprocess.Popen(
        cmd,
        stdout=subprocess.PIPE,
        stderr=subprocess.PIPE,
        shell=True
    )
    out, error = proc.communicate()
    if proc.returncode == 1:
        die(CHECK_NAME, 2, "Can't get ya.subr settings: %s" % error)
    vlan, default_router = out.split(";")
    return {
        "fb_vlan": vlan,
        "fb_default_router": default_router
    }


def check_ya_subr():
    settings = get_ya_subr_settings()
    # NB: fb_default_router can be equal to XXXX
    if settings["fb_default_router"] != "":
        ping_all()
    else:
        die(CHECK_NAME, 2, "Can't determine def router")


def main():
    ip = ''
    if os.path.exists(RC_CONF_LOCAL):
        with open(RC_CONF_LOCAL) as f:
            for line in f:
                line = line.strip()
                parts = line.split('=', 1)
                if len(parts) != 2:
                    continue
                k, v = parts
                if k == 'ipaddr':
                    ip = v  # read args without \n
                    ip = ip.replace('"', '')
    if ip == '':
        if os.path.exists(NETWORKS_MAP_FILE):
            check_networks_json()
        else:
            die(CHECK_NAME, 1,
                "No ipaddr defined in %s and not found %s" % (RC_CONF_LOCAL, NETWORKS_MAP_FILE))
    else:
        if os.path.exists(YA_SUBR):
            check_ya_subr()
        elif os.path.exists(YA_SUBR_JSON):
            check_ya_subr_json()
        else:
            die(CHECK_NAME, 1, "Can't determine networks and gates")


if __name__ == '__main__':
    main()
