#!/usr/bin/env python
# -*- coding: utf-8 -*-
# Provides: grub2_check
# See: https://wiki.yandex-team.ru/runtime-cloud/juggler-bundle/#grub2-check

import os
import subprocess
import time
import re
import fcntl

CHECK_NAME = 'grub2_check'


def die(check_name, status, message):
    print 'PASSIVE-CHECK:%s;%s;%s' % (check_name, status, message)
    raise SystemExit(0)


def readline_from_file(f_name, max_size=1024):
    try:
        with open(f_name) as f:
            return f.readline(max_size).strip()
    except Exception as e:
        die(CHECK_NAME, 1, 'failed to read "{}": {}'.format(f_name, e))


def list_disks():
    disks = {}
    for name in os.listdir('/sys/block/'):
        if not name.startswith('sd'):
            continue
        vendor = readline_from_file('/sys/block/{}/device/vendor'.format(name))
        if not vendor.startswith('ATA'):
            continue
        rotational = readline_from_file('/sys/block/{}/queue/rotational'.format(name))
        if rotational.startswith('0'):
            disks.update({name: {'ssd': True}})
        else:
            disks.update({name: {'ssd': False}})
    return disks


def grub_review():
    disks = list_disks()
    grub = {}
    hdd = 0
    for d in disks.itervalues():
        hdd += 1 if not d['ssd'] else 0
    if hdd:
        for d in disks.iterkeys():
            if not disks[d]['ssd']:
                grub.update({d: check_grub2(d)})
    else:
        for s in disks.iterkeys():
            grub.update({s: check_grub2(s)})
    return grub


def check_grub2(disk):
    grub = subprocess.call('sudo /bin/dd if=/dev/%s bs=512 count=1 2>/dev/null | grep -aq GRUB' % disk, shell=True)
    return not grub


def estimator(grub):
    bad_dsk = []
    exit_num = 0
    for disk, g in grub.iteritems():
        if not g:
            bad_dsk.append(disk)
    if bad_dsk:
        exit_num = 2
    return exit_num, 'Grub2 missing on these disks: %s' % bad_dsk


def grub_probe():
    max_timeout = 5
    timeout_step = 0.1

    err_re = re.compile(r'grub-probe: error: disk `(\S+)\' not found.')
    with open(os.devnull, 'w') as null:
        probe = subprocess.Popen(['sudo', '-n', '/usr/sbin/grub-probe', '-t', 'abstraction', '/'], stderr=subprocess.PIPE, stdout=null)
    timeout = 0
    while probe.poll() is None:
        if timeout > max_timeout:
            break
        timeout += timeout_step
        time.sleep(timeout_step)
    else:
        if probe.poll() != 0:
            _, err = probe.communicate()
            err_substr = err_re.search(err)
            if err_substr:
                return (2, 'grub-probe error on disk {}'.format(err_substr.group(1)))
            else:
                return (1, 'grub-probe other error: {}'.format(err))
        else:
            return (0, None)
    return (1, 'grub-probe execution error')


def main():
    output_msg = []
    errlvl = 0
    exist_grub = grub_review()
    exit_num, msg = estimator(exist_grub)
    if exit_num != 0:
        output_msg.extend([msg, ])
        errlvl = max(errlvl, exit_num)

    exit_num, msg = grub_probe()
    if exit_num != 0:
        output_msg.extend([msg, ])
        errlvl = max(errlvl, exit_num)

    if errlvl == 0:
        output_msg = 'Ok'
    else:
        output_msg = ' '.join(output_msg)

    die(CHECK_NAME, errlvl, output_msg)


if __name__ == '__main__':
    try:
        lock_file = open('/dev/shm/grub2_check', 'w')
        fcntl.flock(lock_file, fcntl.LOCK_EX | fcntl.LOCK_NB)
    except Exception as e:
        die(CHECK_NAME, 2, 'Failed to acquire lock: {}'.format(e))
    else:
        main()
