#!/usr/bin/env python
# -*- coding: utf-8 -*-
# copy from load_monitor.py
# Checks:
# 1. Free Space on /place <20% - yellow, < 10% - red
# 2. Disk Utilization > 80% time
# 3. Network load > 80%
# 4. CPU load > 70%
#
# Provides: load_monitor

import os
import time
import sys

CHECK_NAME = 'load_monitor'

fs_red = 90
fs_yellow = 80
db_red = 50
net_red = 80
cpu_red = 97
io_red = 30

WAIT_TIME = 10


class Iface(object):
    def __init__(self, name, speed):
        self.name = name
        self.speed = speed


def die(n, s):
    print("PASSIVE-CHECK:%s;%d;%s" % (CHECK_NAME, n, s))
    raise SystemExit(0)


def read_int_from_file(f_name, max_size=1024):
    try:
        with open(f_name) as f:
            return int(f.readline(max_size).strip())
    except Exception as e:
        return 0


def list_ifaces():
    ifs = []
    ifaces = []
    for iface in os.listdir("/sys/class/net/"):
        if iface.startswith(('lo', 'tun', 'vla')):
            continue
        ifs.append(iface)
    for i in ifs:
        try:
            with open("/sys/class/net/" + i + "/speed", "r") as f:
                speed = int(f.readline().strip())
                if speed != 4294967295 and speed != 65535:
                    ifaces.append(Iface(i, speed))
        except Exception:
            continue
    return ifaces


def get_link_stats(links):
    rv = {}
    for l in links:
        name = l.name
        rv[name] = {
            'rx': read_int_from_file("/sys/class/net/" + name + "/statistics/rx_bytes"),
            'tx': read_int_from_file("/sys/class/net/" + name + "/statistics/tx_bytes")
        }
    return rv


def cpu_stat():
    try:
        with open("/proc/stat") as f:
            return [int(i) for i in f.readline().strip().split()[1:]]
    except:
        return []


def main():
    v = True if len(sys.argv) > 1 and sys.argv[1] == "-v" else False

    err = 0
    msg = ""
    # Gather interfaces
    ifaces = list_ifaces()
    # Get stats, sleep, get again
    link = get_link_stats(ifaces)
    cpu = cpu_stat()
    time.sleep(WAIT_TIME)
    link_after = get_link_stats(ifaces)
    cpu_after = cpu_stat()
    for l in ifaces:
        name = l.name
        # Mbit/s
        rx = float(link_after[name]["rx"] - link[name]["rx"]) * 8 / 1024 / 1024 / WAIT_TIME
        tx = float(link_after[name]["tx"] - link[name]["tx"]) * 8 / 1024 / 1024 / WAIT_TIME
        rp = rx / l.speed * 100
        tp = tx / l.speed * 100
        if rp > net_red or tp > net_red or v:
            err = 2
            msg += "|%s:rx/tx=%.2f/%.2f/%dMbit/s" % (l.name, rx, tx, l.speed)
    # % of time all cores are busy
    cpu_diff = [j - cpu_after[l] for l, j in enumerate(cpu)]
    c = float(cpu_diff[0] + cpu_diff[1] + cpu_diff[2]) * 100 / sum(cpu_diff)
    io = float(cpu_diff[4]) * 100 / sum(cpu_diff)
    if c > cpu_red or io > io_red or v:
        err = 2
        msg += "|cpu:%.2f%%|io:%.3f%%" % (c, io)
    if err == 0 and not v:
        msg = "|Ok"
    elif v:
        err = 0
    die(err, msg[1:])


if __name__ == '__main__':
    main()
