#!/usr/bin/env python
# -*- coding: utf-8 -*-
# Provides: nvidia_containers

import os
import sys
import subprocess
import re
import fcntl

nvdevpattern = r'^nvidia\d+$'
nvdevchk = re.compile(nvdevpattern)


def list_nvidia():
    devlist = []  # nvidia devices
    nvcount = 0  # /dev/nvidia, to
    for fl in os.listdir('/dev'):
        if os.major(os.lstat('/dev/{}'.format(fl)).st_rdev) == 0xc3:
            if nvdevchk.match(fl):
                nvcount += 1
            devlist.append('/dev/{}'.format(fl))
    return devlist, nvcount


def run_test_containers(cmd):
    '''
    :cmd: Command to run in subcontainer. Str

    Run test subcontainers from containers with state in :states_list: and returns
    tuple with lists of  containers with successful runs [0],
    runs unsuccessful [1], and with system errors[2]
    '''
    ok_list = []
    err_list = []
    sysfail_list = []
    try:
        sout = subprocess.Popen(['portoctl', 'list', '-r', '-1'], stdout=subprocess.PIPE).communicate()[0]
        containers = set((l.split('/')[0] for l in sout.strip().split('\n')))
    except:
        return ([], [], ['portoctl communication error'])
    try:
        with open(os.devnull, 'w') as devnull:
            for container in containers:
                ret = subprocess.call(['sudo', 'portoctl', 'exec', '{}/Juggler_check_nvidia_containers'.format(container), cmd], stdout=devnull)
                if ret == 0:
                    ok_list.append(container)
                else:
                    err_list.append(container)
    except:
        sysfail_list.append(container)
    return (ok_list, err_list, sysfail_list)


def die(name, status, msg):
    print("PASSIVE-CHECK:%s;%s;%s" % (name, status, msg))


def main():
    try:
        lock_file = open('/dev/shm/nvidia_check.lock', 'w')
        fcntl.flock(lock_file, fcntl.LOCK_EX | fcntl.LOCK_NB)
    except Exception as e:
        die('nvidia_containers',
            2,
            'Failed to acquire lock: {}'.format(e))
        sys.exit(2)

    nvlist, dev_cnt = list_nvidia()

    check_cmd = "command=ls -1 {}".format(' '.join(nvlist))
    cnt = run_test_containers(check_cmd)
    if cnt[2]:
        die('nvidia_containers',
            2,
            "SysError while test containers: {}".format(', '.join(cnt[2])))
    elif cnt[1]:
        die('nvidia_containers',
            2,
            "Containers without NVidia devs: {}".format(', '.join(cnt[1])))
    else:
        die('nvidia_containers',
            0,
            "All NVidia devices passed to containers")


if __name__ == '__main__':
    main()
