#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# Provides: skybone_locked_mem
# Rationale:
# If someone mlocks files which were downloaded by skybone
# this memory will be accounted to skybone (if those pages where in skybone's page cache).
# Yandex kernel tries to move pages upon mlock if target cgroup has
# larger memory limit.
# See: https://wiki.yandex-team.ru/runtime-cloud/juggler-bundle/#skybone-locked-mem

import errno


CHECK_NAME = "skybone_locked_mem"
SKYBONE_CGROUP = '/sys/fs/cgroup/memory/skynet/services/skybone/'
STAT_FILE = SKYBONE_CGROUP + 'memory.stat'
LIMIT_FILE = SKYBONE_CGROUP + 'memory.limit_in_bytes'

MAX_LOCKED_RATIO = 0.50  # Empirically chosen value


def exit(code, msg):
    print "PASSIVE-CHECK:{};{};{}".format(CHECK_NAME, code, msg)
    raise SystemExit(0)


def ok():
    exit(0, "OK")


def fail(msg):
    exit(2, msg)


def read_locked():
    try:
        with open(STAT_FILE, 'rb', 4096) as f:
            for l in f:
                if l.startswith('total_unevictable'):
                    # E.g. total_unevictable 178225152
                    locked_bytes = int(l[l.find(' '):])
                    return locked_bytes
    except EnvironmentError as e:
        if e.errno == errno.ENOENT:
            # If no skybone memory cgroup - nothing is locked
            return 0
        raise


def read_limit():
    try:
        with open(LIMIT_FILE, 'rb', 4096) as f:
            return int(f.read(512))
    except EnvironmentError as e:
        if e.errno == errno.ENOENT:
            # If no limit - zero is fine
            return 0
        raise


def main():
    locked = read_locked()
    if not locked:
        ok()
    limit = read_limit()
    if not limit:
        ok()
    ratio = locked / float(limit)
    if ratio > MAX_LOCKED_RATIO:
        fail('{:.5f} skybone memory is locked'.format(ratio))
    return ok()


if __name__ == '__main__':
    main()

