"""
Common utils for rtc juggler bundle.

"""
# Ensure you have added tests and docstrings for new stuff! Pleeease =)

import fcntl
import json
import os
import psutil
import sys
import time

from juggler.bundles import Event

LOCKS_DIR = '/dev/shm/'


class WalleCompatEvent(Event):
    """
    Wall-e compatible juggler event.

    By Wall-e compatible means JSON object with certain fields as description.

    """

    def __init__(self, status, description, timestamp=None, **kwargs):
        description = json.dumps({
            'status': status.name,
            'timestamp': int(time.time()) if timestamp is None else timestamp,
            'reason': truncate_description_string(description),
        })
        super().__init__(status, description, **kwargs)


def get_file_content(path):
    """
    Return content of a file.

    """
    with open(path) as f:
        return f.read().strip()


def get_process_info(pid, attrs=('name', 'create_time', 'status')):
    """
    Return dict with process info.

    See https://psutil.readthedocs.io/en/latest/#psutil.Process.as_dict for all
    available attributes.

    """
    return psutil.Process(pid=pid).as_dict(attrs=attrs)


class flock(object):
    """
    flock wrapper, may be used as context manager or as standalone object

    Locks created by flock are preserved across an execve, so any child process
    will inherite it.

    """
    def __init__(self, fd=None, mode=fcntl.LOCK_EX | fcntl.LOCK_NB,
                 directory=LOCKS_DIR, keep_file=False):
        if fd is None:
            self.fd = open(directory + os.path.basename(sys.argv[0]), 'w')
            self.keep_file = keep_file
        else:
            self.fd = fd
            self.keep_file = True

        self.mode = mode

    def __enter__(self):
        self.lock()
        return self

    def __exit__(self, type, value, traceback):
        if self.keep_file:
            self.unlock()
        else:
            os.unlink(self.fd.name)

    def lock(self):
        fcntl.flock(self.fd, self.mode)

    def unlock(self):
        fcntl.flock(self.fd, fcntl.LOCK_UN)


def truncate_description_string(desc, maxlen=700):
    """
    truncate_description_string limit description length to maxlen

    :param desc: Original description
    :param maxlen: Max description length
    :return: returns a description limited to maxlen
    TODO: fair sting bytes length may be needed
    """
    shorten_str = ' ... '

    if len(desc) > maxlen:
        idx = int(maxlen/2) - int(len(shorten_str)/2)
        return desc[:idx] + shorten_str + desc[(len(desc) - idx + 1):]

    else:
        return desc
