"""
Juggler checks builder for Salt masters.

"""
import logging

from infra.reconf import ConfOptDenied, NodesHandler

from infra.reconf_juggler.builders import JugglerChecksBuilder
from infra.reconf_juggler.opts import aggregators
from infra.reconf_juggler.trees import NodeSet

from infra.rtc.juggler.reconf.checks import hostman as hostman_checks
from infra.rtc.juggler.reconf.checksets import RtcCheckSetProd
from infra.rtc.juggler.reconf.optfactories import RtcOptFactory
from infra.rtc.juggler.reconf.opts import aggregators as rtc_aggregators

import infra.rtc.juggler.reconf.builders.projects.salt_masters.checks as salt_masters_checks


class LeafAggregator(aggregators.TimedMoreThanLimitIsProblem):
    quorum_calc = None  # every host important


class ProductionAggregator(rtc_aggregators.Production):
    leaf_aggregator = LeafAggregator


class SaltMastersNodesHandler(NodesHandler):
    # Devel masters are powered by YP, so don't create host aggregates for such
    # instances (HOSTMAN-113)
    def get_handled_value(self, key, val):
        if val is not None and val.origin == 'salt_masters_devel':
            if val.__class__ is not salt_masters_checks.hostman_server:
                logging.info('Discard ' + key + ' (irrelevant, HOSTMAN-113)')
                raise ConfOptDenied

        return super().get_handled_value(key, val)


class OptFactory(RtcOptFactory):
    """
    Inject specific opt handlers.

    """
    def create_handler(self, handler_class, check):
        if issubclass(handler_class, rtc_aggregators.Production):
            return ProductionAggregator(bound=check)

        if issubclass(handler_class, NodesHandler):
            return SaltMastersNodesHandler(bound=check)

        return super().create_handler(handler_class, check)


class SaltMastersCheckSet(RtcCheckSetProd):
    branches = (
        *RtcCheckSetProd.branches,
        hostman_checks.system_service_yandex_hm_reporter,
        salt_masters_checks.hostman_server,
        salt_masters_checks.hostman_server_emergency,
    )


class Builder(JugglerChecksBuilder):
    def build_initial_tree(self, initial_data):
        return NodeSet(
            {
                'salt_masters': {
                    'children': {
                        'devel': {
                            'children': {'SKYLH%yp@salt-master-prestable': None},
                            'tags': ['grp_devel'],
                        },
                        'man': {
                            'children': {'WALLE%PROD@prj=rtc-mtn-hostman&datacenter=man': None},
                            'tags': ['grp_man'],
                        },
                        'msk': {
                            'children': {'WALLE%PROD@prj=rtc-mtn-hostman&datacenter=myt': None},
                            'tags': ['grp_msk'],
                        },
                        'sas': {
                            'children': {'WALLE%PROD@prj=rtc-mtn-hostman&datacenter=sas': None},
                            'tags': ['grp_sas'],
                        },
                        'vla': {
                            'children': {'WALLE%PROD@prj=rtc-mtn-hostman&datacenter=vla': None},
                            'tags': ['grp_vla'],
                        },
                    },
                    'tags': ['prj_salt-masters'],
                },
            },
        )

    def build_checks_tree(self, initial_tree):
        return SaltMastersCheckSet(
            initial_tree,
            opt_factory=OptFactory(),
            resolver=self.resolver,
        )


def main():
    Builder().run(app_desc=__doc__)
