"""
Juggler checks builder for RTC infra experiment cluster.
https://st.yandex-team.ru/RTCNETWORK-235

"""
from infra.reconf_juggler.opts import aggregators, notifications
from infra.reconf_juggler import CheckSet
from infra.reconf_juggler.builders import JugglerChecksBuilder
from infra.rtc.juggler.reconf.checks import (
    hostman,
    infrastructure,
    network,
    cloud,
    unreachable,
)
import infra.rtc.juggler.reconf.builders.projects.sysdev_experiment.checks as sysdev_experiment_checks
from infra.rtc.juggler.reconf.optfactories import RtcOptFactory
from infra.rtc.juggler.reconf.opts import (
    aggregators as rtc_aggregators,
    notifications as rtc_notifications,
)


RTC_TESTING = {
    'WALLE%PROD@prj=rtc-testing-mtn': None,
    'WALLE%PROD@prj=rtc-testing': None
}
SANDBOX = {'rtc-sysdev-sandbox': None}
L3 = {'rtc-infra-sysdev-slb.yandex-team.ru': None}
NUMA_MEMBW = {'WALLE%PROD@prj=yp-iss-vla': None}


class Checks(CheckSet):
    branches = (
        hostman.hostman_salt,
        hostman.hostman_apt,
        infrastructure.ssh,
        infrastructure.bind,
        cloud.check_iss_agent,
        cloud.portod_tasks,
        network.abnormal_ip_conf,
        network.check_ra,
        network.fb_route_check,
        network.hbf_agent_rtc,
        network.ip_groups,
        network.tsolro_check,
        network.net_irq,
        unreachable.META,
        unreachable.UNREACHABLE,
        unreachable.bundle_info,
    )


class SandboxChecks(CheckSet):
    branches = (
        sysdev_experiment_checks.reboot_scheduler,
    )


class L3Checks(CheckSet):
    branches = (
        sysdev_experiment_checks.l3_ssh_check,
    )


class NumaMembwCheck(CheckSet):
    branches = (
        sysdev_experiment_checks.numa_memorybw,
    )


class SysdevLeafAggregator(rtc_aggregators.ProductionLeafAggregator):
    """
    SYSDEV Aggregator for leaf checks.

    """
    nodata_mode = 'skip'

    def get_default_limits(self):
        return [
            {
                'crit': '100%',
                'day_end': 5,
                'day_start': 1,
                'time_end': 9,
                'time_start': 0,
                'warn': '0'
            },
            {
                'crit': '100%',
                'day_end': 7,
                'day_start': 6,
                'time_end': 23,
                'time_start': 0,
                'warn': '0'
            },
            {
                'crit': '0',
                'day_end': 5,
                'day_start': 1,
                'time_end': 23,
                'time_start': 9,
                'warn': '100%'
            },
        ]


class SysdevAggregator(rtc_aggregators.Production):
    """
    Return SYSDEV aggregator depending on tree level.

    """
    leaf_aggregator = SysdevLeafAggregator


class SysdevOptFactory(RtcOptFactory):
    def create_handler(self, handler_class, check):
        if issubclass(handler_class, aggregators.AggregatorHandler):
            return super().create_handler(SysdevAggregator, check)
        elif issubclass(handler_class, notifications.NotificationsHandler):
            return super().create_handler(rtc_notifications.Sysdev, check)

        return super().create_handler(handler_class, check)


class Builder(JugglerChecksBuilder):
    def build_checks_tree(self, initial_tree):
        result = CheckSet()
        result.update(Checks({'sysdev_experiment': {'children': RTC_TESTING}}, opt_factory=SysdevOptFactory()))
        result.update(SandboxChecks({'sysdev_experiment': {'children': SANDBOX}}, opt_factory=SysdevOptFactory()))
        result.update(L3Checks({'sysdev_experiment': {'children': L3}}, opt_factory=SysdevOptFactory()))
        result.update(NumaMembwCheck({'sysdev_experiment': {'children': NUMA_MEMBW}}, opt_factory=SysdevOptFactory()))
        return result


def main():
    Builder().run(app_desc=__doc__)
