"""
Juggler aggregates builder for HBF drops (RTCNETWORK-390).

"""
from infra.reconf.util.symbols import symbol_full_name
from infra.reconf_juggler.builders import DeclaredChecksBuilder

from infra.rtc.juggler.reconf.opts import notifications
from infra.rtc.juggler.reconf.checks.network import AbstractNetworkCheck


REFRESH_TIME = 5
NAMESPACE = 'rtcsys'

YASM_SIGNALS_CLUSTERS = ('asearch', 'qloud')
YASM_SIGNALS_LOCATIONS = ('sas', 'vla', 'man', 'msk')
YASM_SIGNALS_DIRECTIONS = ('input', 'output')
YASM_SIGNALS_INTERVALS = (
    '1_60',
    '60_600',
    '600_6000',
    '6000_60000',
    '60000_inf',
)

COMMON_TAGS = [
    'hbf',
    'marty',
    'mondata_help_rtchbfdrops',
    'is_disaster',
]

NOTIFY = {
    'description': 'HBF drops hcount per DC aggregated',
    'template_name': 'on_status_change',
    'template_kwargs': {
        'login': ['rtc-networking', 'Runtime_Monitoring_HBF'],
        'method': 'telegram',
        'status': [
            'CRIT',
            'WARN',
        ],
    },
}


class HbfDropsNotifications(notifications.EssentialNotifications):
    notify_rules = notifications.EssentialNotifications.notify_rules + (NOTIFY,)


class HbfDropsCheck(AbstractNetworkCheck):
    """ HBF drops check """
    doc_url = 'https://st.yandex-team.ru/RTCNETWORK-390'
    _notifications = HbfDropsNotifications


class Builder(DeclaredChecksBuilder):
    default_check_class = symbol_full_name(HbfDropsCheck)

    def get_node(self, cluster, location, direction):
        node = {
            'children': {},
            'meta': {
                'urls': [
                    {
                        'title': 'Yasm panel',
                        'type': 'yasm_panel',
                        'url': (
                            'https://yasm.yandex-team.ru/template/panel/' +
                            'runtimecloud_hbf_drops/hosts=' + cluster.upper() +
                            ';geo=' + location + '/?range=7200000'
                        ),
                    },
                    {
                        'title': 'Top drops chart',
                        'type': 'solomon',
                        'url': (
                            'https://solomon.yandex-team.ru/?'
                            'project=hbf-per-service-drops'
                            '&cluster=production'
                            '&service=srv'
                            '&graph=production_top10_host_cluster_IN'
                            '&b=1h'
                            '&e='
                        ),
                    },
                    {
                        'title': 'Top reject chart',
                        'type': 'solomon',
                        'url': (
                            'https://solomon.yandex-team.ru/?'
                            'project=hbf-per-service-drops'
                            '&cluster=production'
                            '&service=srv'
                            '&graph=production_top10_host_cluster_OUT'
                            '&b=1h'
                            '&e='
                        ),
                    },
                ],
            },
            'namespace': NAMESPACE,
            'tags': COMMON_TAGS + [cluster],
            'refresh_time': REFRESH_TIME,
        }

        for interval in YASM_SIGNALS_INTERVALS:
            service = f'{cluster}_hbf_drops_geowide_{location}_{direction}_{interval}'
            origin = 'yasm_' + service
            node['children'][origin + ':' + service] = {}

        name = (
            f'hbf.drops.hcount.{location}:'
            f'hbf_drops_{cluster}_hostcount_geowide_{location}_{direction}'
        )

        return name, node

    def build_initial_tree(self, initial_data):
        nodes = {}

        for cluster in YASM_SIGNALS_CLUSTERS:
            for location in YASM_SIGNALS_LOCATIONS:
                for direction in YASM_SIGNALS_DIRECTIONS:
                    name, node = self.get_node(cluster, location, direction)
                    nodes[name] = node

        return nodes


def main():
    Builder().run(app_desc=__doc__)
