""" Sysdev overall aggregates builder (HOSTMAN-484) """

from infra.reconf_juggler import CheckSet
from infra.reconf_juggler.builders import JugglerChecksBuilder
from infra.reconf_juggler.opts.notifications import NotificationsHandler
from infra.reconf_juggler.trees.locations import Locations

from infra.reconf_juggler.opts import aggregators
from infra.rtc.juggler.reconf.opts import aggregators as rtc_aggregators

from infra.rtc.juggler.reconf.checks import (
    network as net_checks,
    unreachable as unreach_checks,
)
from infra.rtc.juggler.reconf.optfactories import RtcOptFactory
from infra.rtc.juggler.reconf.opts import notifications


class SysdevOverallChecksSet(CheckSet):
    branches = (
        unreach_checks.UNREACHABLE,
        net_checks.hbf_rules_validation,
    )


class SysdevOverallLeafAggregator(aggregators.TimedMoreThanLimitIsProblem):

    allow_empty_groups = True

    def get_default_limits(self):
        return [
            {
                'day_start': 1,
                'day_end': 5,
                'time_start': 0,
                'time_end': 9,
                'warn': '2%',
                'crit': '3%',
            },
            {
                'day_start': 1,
                'day_end': 5,
                'time_start': 9,
                'time_end': 23,
                'warn': '1%',
                'crit': '2%',
            },
            {
                'day_start': 6,
                'day_end': 7,
                'time_start': 0,
                'time_end': 23,
                'warn': '2%',
                'crit': '3%',
            },
        ]


class SysdevOverallProductionAggregator(rtc_aggregators.Production):
    leaf_aggregator = SysdevOverallLeafAggregator


class SysdevOverallOptFactory(RtcOptFactory):
    def create_handler(self, handler_class, check):
        if issubclass(handler_class, rtc_aggregators.Production):
            return SysdevOverallProductionAggregator(bound=check)
        if issubclass(handler_class, NotificationsHandler):
            if check.__class__ is unreach_checks.UNREACHABLE:
                return super().create_handler(
                    notifications.EssentialNotifications, check)

        return super().create_handler(handler_class, check)


class Builder(JugglerChecksBuilder):
    def build_initial_data(self, unused):
        resolved = self.resolver['walle']['projects'].resolve({
            'tags': ('rtc',),
            'fields': ('id',),
        })

        projects = {}
        for prj in resolved:
            projects['WALLE%PROD@prj=' + prj['id']] = None

        return projects

    def build_initial_tree(self, initial_data):
        return Locations(
            {'sysdev_overall': {'children': initial_data}},
            resolver=self.resolver,
        )

    def build_checks_tree(self, initial_tree):
        return SysdevOverallChecksSet(
            initial_tree,
            opt_factory=SysdevOverallOptFactory(),
            resolver=self.resolver,
        )


def main():
    Builder().run(app_desc=__doc__)
