"""
Wall-e hosts status aggregates builder.
Early prototype, https://st.yandex-team.ru/WALLE-3766

Aggregates build over event groups for two main reasons:
    1. Delegate hosts grouping logic to wall-e/it's users.
    2. Avoid huge aggregates (aggregate only crits and warns selecting by
       desired tags)

See https://docs.yandex-team.ru/juggler/aggregates/groups#events for details
about events groups.

# EVENTS REQUIREMENTS
1. Wall-e should emit `walle_status` event for each host.
2. Events must be tagged accordingly requested aggregates.
3. Events should be emitted as soon as possible for status/state change to
   maintain actual aggregates and raw events status.

4. Event status MUST be one of:
   OK - host is ready for production use.
   WARN - host is moving from OK to CRIT (waiting for CMS, for example).
   CRIT - host is not suitable for production use.

See https://st.yandex-team.ru/WALLE-3801 for details about events.

"""
from infra.reconf_juggler.builders import JugglerChecksBuilder

from infra.rtc.juggler.reconf.checks.walle import walle_status
from infra.rtc.juggler.reconf.checksets import WalleStatusCheckSet
from infra.rtc.juggler.reconf.optfactories import RtcOptFactory
from infra.rtc.juggler.reconf.opts import aggregators, ttls

from infra.rtc.juggler.reconf.builders.projects.walle.events import get_causes_list


def fetch_aggregation_parameters_from_walle():
    causes_list = get_causes_list()  # just to avoid copy/paste, not a default
    times_list = [  # noqa
        # 'lt1d',
        # 'gt3d',
        'gt7d',
        # 'gt14d',
    ]

    # Return desired aggregates structure. Some parts are commented due lack of
    # duration data (aggregates will be in NO DATA state w/o it).
    return [
        # Groups are mostly for projects owners to seek and heal dead hosts.
        # Here we allow projects owners to define their own aggregates
        # structure and options. Also here we can provide most common structure
        # as default when group created.
        {
            'type': None,  # 'None' means intermediate aggregate (no tags used)
            'items': ['by_group'],  # name for it
            'aggregate': [
                {
                    'type': 'group',  # tag key (any type tag, see WALLE-3801)
                    'items': [  # tag values
                        'rtc',
                        'rtc-yp-gpu-none',
                        'rtc-yp-gpu-nvidia',
                        'rtc-yt-gpu-none',
                        'rtc-yt-gpu-nvidia',
                    ],
                    'aggregate': [  # how to buld subaggregates
                        {
                            'type': 'status',
                            'items': ['dead'],  # btw: here may be any statuses
                            'aggregate': [
                                {
                                    'type': 'cause',
                                    'items': causes_list,
                                    # 'aggregate': [
                                    #    {
                                    #        'type': 'time',
                                    #        'items': times_list,
                                    #    },
                                    # ],
                                },
                                {  # override checks_missing aggregate opts
                                    'type': 'cause',
                                    'items': ['checks_missing'],
                                    # 'opts': {
                                    #    'notifications': [
                                    #        {
                                    #            'template_name': 'on_status_change',
                                    #            'template_kwargs': {
                                    #                'login': 'Search_Runtime',
                                    #                'method': 'telegram',
                                    #                'repeat': 14400,
                                    #                'status': [
                                    #                    'CRIT',
                                    #                ],
                                    #            },
                                    #        },
                                    #    ],
                                    # },
                                },
                            ],
                        },

                    ],
                },
            ],
        },

        # Mostly for wall-e SRE to track causes and their impact
        {
            'type': None,  # grouping aggregste
            'items': ['by_cause'],
            'aggregate': [
                {
                    'type': 'cause',
                    'items': causes_list,
                    'aggregate': [
                        {
                            'type': 'status',
                            'items': ['dead'],
                            # 'aggregate': [
                            #    {
                            #        'type': 'time',
                            #        'items': times_list,
                            #    },
                            # ],
                        },
                    ],
                },
            ],
        },

        # Useful for wall-e SRE to haunt rotten hosts
        # {
        #    'type': None,
        #    'items': ['by_duration'],
        #    'aggregate': [
        #        {
        #            'type': 'time',
        #            'items': ['gt3d', 'gt7d', 'gt14d'],
        #            'aggregate': [
        #                {
        #                    'type': 'status',
        #                    'items': ['dead'],
        #                },
        #            ],
        #        },
        #    ],
        # },
    ]


class WalleLeafAggregator(aggregators.ProductionLeafAggregator):
    downtimes_mode = 'ignore'
    nodata_mode = 'skip'
    quorum_calc = None

    def get_default_limits(self):
        return [
            {
                'crit': '3',
                'day_end': 5,
                'day_start': 1,
                'time_end': 23,
                'time_start': 0,
                'warn': '1',
            },
        ]


class WalleAggregator(aggregators.Production):
    leaf_aggregator = WalleLeafAggregator


class WalleOptFactory(RtcOptFactory):
    def create_handler(self, handler_class, check):
        if issubclass(handler_class, aggregators.Production):
            return super().create_handler(WalleAggregator, check)

        return super().create_handler(handler_class, check)


class Hosts(walle_status):
    """ Wall-e hosts status """
    doc_url = walle_status.doc_url  # TODO: set actual link when ready

    _ttl = ttls.FortyFiveMinutes

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

        if self.has_endpoints():
            for key, val in tuple(self['children'].items()):
                if val is None:
                    del self['children'][key]
                    selector = 'service=walle_status' + key

                    # DIRTY, DIRTY, DIRTY hack.
                    # We build aggregates over CRIT events only. This means
                    # events absence is OK (our goal is to eliminate crits).
                    # But right now it is impossible to configure aggregate to
                    # be OK when no one event resolved by EVENTS selector. So,
                    # to avoid `CHILD_GROUP_INVALID/All children groups are
                    # empty/not cached` states we simply add some stable event
                    # to selector.
                    # Sure this will pollute resulting aggregates.
                    # But since this event is in stable OK will be just a noise
                    # on the charts (and yes, in questions WTF, why such a
                    # thing doing here).
                    selector = '({})|({})'.format(
                        selector,
                        'host=good.yandex.ru&service=walle_tor_link',
                    )

                    self['children']['EVENTS%' + selector] = None

    @classmethod
    def provides(cls, is_node=True):
        if is_node:
            return 'hosts'

        return None


class WalleEventsCheckSet(WalleStatusCheckSet):
    branches = (Hosts,)


class EventTag(object):
    def __init__(self, key, val):
        self.key = key
        self.val = val

    def __str__(self):
        return self.key + '-' + self.val


class Builder(JugglerChecksBuilder):
    def get_aggregates(self, rules, prefix='', events_tags=()):
        aggregates = {}

        for rule in rules:
            if not isinstance(rule['items'], (list, tuple)):
                # just to save debug time when str passes
                raise TypeError('list or tuple expected for items, got ' +
                                rule['items'].__class__.__name__)

            for item in rule['items']:
                if rule['type'] is None:  # intermediate (grouping) aggregate
                    origin = prefix + item
                    current_events_tags = events_tags
                else:
                    origin = prefix + rule['type'] + '_' + item
                    current_events_tags = events_tags + (EventTag(rule['type'], item),)

                tags = [str(x) for x in current_events_tags]

                if 'aggregate' in rule:
                    children = self.get_aggregates(
                        rule['aggregate'],
                        prefix=prefix if rule['type'] is None else origin + '_',
                        events_tags=current_events_tags,
                    )
                else:
                    children = {'&tag=' + '&tag='.join(tags): None}

                # Introduce 'aggregate sequence tag' to be able to select exact
                # aggregates in dashboards and UI filters
                aggregate_tag = '-'.join(x.key for x in current_events_tags)
                if aggregate_tag:
                    tags.append('aggregates-' + aggregate_tag)

                body = rule.get('opts', {})
                body.update(children=children, tags=tags)

                aggregates[origin] = body

        return aggregates

    def build_initial_data(self, unused):
        return fetch_aggregation_parameters_from_walle()

    def build_initial_tree(self, initial_data):
        return {
            'walle': {
                'children': self.get_aggregates(initial_data, prefix='walle_'),
            }
        }

    def build_checks_tree(self, initial_tree):
        return WalleEventsCheckSet(
            initial_tree,
            opt_factory=WalleOptFactory(),
            resolver=self.resolver,
        )


def main():
    Builder().run(app_desc=__doc__)
