"""
Temporary events emitter for wall-e hosts statuses (WALLE-3766, WALLE-3767)

# DISCLAIMER:
Checks used for causes just because we have no other values at the moment.
Cause may be anything what seems reasonable to Wall-e and fit tag names.

# KNOWN ISSUES
* PXE boot problems are not indicated in host['health'] so they are simply
  tagged as cause-unreachable, cause-ssh
* There are no info in host status about duration, so no such tags emitted.

"""
import logging
import os
import requests

from infra.reconf_juggler.resolvers import WalleHostsResolver

JUGGLER_PUSH_URL = os.environ.get('JUGGLER_PUSH_URL', 'http://localhost:31579/events')
MAX_EVENTS_BATCH = 100  # current limit in Juggler API
REQUESTS_SESSION = requests.Session()

logging.basicConfig(format='[%(asctime)s] %(levelname)s %(message)s',
                    level=logging.DEBUG)


def fetch_hosts_status(**kwargs):
    return WalleHostsResolver().resolve_query({
        'tags': ['rtc'],  # keep it for a while (emitter is slow and often killed by timeout)
        'fields': ['health', 'name', 'project', 'state', 'status', 'tags'],
        **kwargs,
    })


def format_juggler_event(host):
    event = {
        'description': get_event_description(host),
        'host': host['name'],
        'instance': '',
        'service': 'walle_status',
        'status': 'OK',
        'tags': get_event_tags(host),
    }

    if host['status'] == 'ready':
        pass
    else:
        event['tags'].extend(get_cause_tags(host))

        if host['status'] in {'dead', 'invalid'}:
            event['status'] = 'CRIT'
        else:
            event['status'] = 'WARN'

    return event


def get_causes_list():
    # host_status = tuple(fetch_hosts_status(limit=1, status='ready'))[0]
    # return list(host_status['health']['check_statuses'].keys())

    return ['bmc', 'certman', 'check_iss_agent', 'check_skynet_procs',
            'checks_missing', 'cpu', 'cpu_caches', 'cpu_capping', 'disk',
            'fs_check', 'gpu', 'hostman-distrib', 'hostman-ready', 'link',
            'll_duplicate_addr', 'memory', 'need_reboot_kernel',
            'portod_tasks', 'rack', 'rack_overheat', 'reboots', 'ssh',
            'switch', 'tainted_kernel', 'tor_link', 'unreachable',
            'walle_clocksource', 'walle_firmware', 'walle_fstab',
            'walle_hotfix', 'walle_meta']


def get_cause_tags(host):
    try:
        return tuple('cause-' + x for x in host['health']['decision']['failures'])
    except KeyError:
        return ['cause-unknown']


def get_event_description(host):
    desc = host['state'] + '/' + host['status'] + '.'

    try:
        desc += ' ' + host['health']['decision']['reason']
    except KeyError:
        pass

    return desc


def get_event_tags(host):
    tags = []
    tags.append('project-' + host['project'])
    tags.append('state-' + host['state'])
    tags.append('status-' + host['status'])

    if 'rtc' in host['tags']:
        # To be able to build summarised (whole-RTC) aggregates and charts
        tags.append('group-rtc')

        # https://st.yandex-team.ru/WALLE-3896
        if 'yt' in host['tags']:
            if 'rtc.gpu-nvidia' in host['tags']:
                tags.append('group-rtc-yt-gpu-nvidia')
            else:
                tags.append('group-rtc-yt-gpu-none')
        elif 'yp' in host['tags']:
            if 'rtc.gpu-nvidia' in host['tags']:
                tags.append('group-rtc-yp-gpu-nvidia')
            else:
                tags.append('group-rtc-yp-gpu-none')
    else:
        # To be able to filter unsorted hosts
        tags.append('group-unknown')

    return tags


def emit_juggler_events(events):
    request = {'source': 'WALLE.3767.testing.script', 'events': events}
    REQUESTS_SESSION.post(JUGGLER_PUSH_URL, json=request)
    # TODO: handle response codes here


def main():
    batch = []

    for host in fetch_hosts_status():
        batch.append(format_juggler_event(host))

        if len(batch) == MAX_EVENTS_BATCH:
            emit_juggler_events(batch)
            batch.clear()

    if batch:
        emit_juggler_events(batch)
