"""
Juggler checks builder for YP masters clusters.

"""

from infra.reconf_juggler.builders import JugglerChecksBuilder
from infra.reconf_juggler.opts import aggregators
from infra.reconf_juggler.trees import NodeSet

import infra.rtc.juggler.reconf.checks
from infra.rtc.juggler.reconf.checksets import RtcCheckSetProd
from infra.rtc.juggler.reconf.optfactories import RtcOptFactory
from infra.rtc.juggler.reconf.opts import aggregators as rtc_aggregators
from infra.rtc.juggler.reconf.opts import notifications as rtc_notifications


# TODO: this list should be revised (inherited from previous aggrs builder)
NOTIFABLE_CHECKS = {
    infra.rtc.juggler.reconf.checks.hardware.cpu_throt_capping_check,
    infra.rtc.juggler.reconf.checks.hardware.cpu_throt_check,
    infra.rtc.juggler.reconf.checks.hardware.cpu_throt_frequency_check,
    infra.rtc.juggler.reconf.checks.hardware.cpu_throt_perf_check,
    infra.rtc.juggler.reconf.checks.hardware.cpu_throt_thermal_check,
    infra.rtc.juggler.reconf.checks.hardware.cpu_throt_turboboost_check,

    infra.rtc.juggler.reconf.checks.hardware.walle_bmc,
    infra.rtc.juggler.reconf.checks.hardware.walle_cpu,
    infra.rtc.juggler.reconf.checks.hardware.walle_disk,
    infra.rtc.juggler.reconf.checks.hardware.walle_fs_check,
    infra.rtc.juggler.reconf.checks.hardware.walle_link,
    infra.rtc.juggler.reconf.checks.hardware.walle_memory,
    infra.rtc.juggler.reconf.checks.hardware.walle_tainted_kernel,

    infra.rtc.juggler.reconf.checks.hostman.hostman_salt,

    infra.rtc.juggler.reconf.checks.infrastructure.cron,
    infra.rtc.juggler.reconf.checks.infrastructure.fs_mount_opts,
    infra.rtc.juggler.reconf.checks.infrastructure.ssh,

    infra.rtc.juggler.reconf.checks.cloud.skynet_copier,

    infra.rtc.juggler.reconf.checks.network.abnormal_ip_conf,

    infra.rtc.juggler.reconf.checks.unispace.fs_free_space_syst,

    infra.rtc.juggler.reconf.checks.unreachable.META,
    infra.rtc.juggler.reconf.checks.unreachable.UNREACHABLE,
}

YP_OPS_EMAIL_NOTIFY = {
    'description': 'RUNTIMECLOUD-7626',
    'template_name': 'on_status_change',
    'template_kwargs': {
        'login': 'yp-monitoring',
        'method': 'email',
        'status': [
            'CRIT',
        ],
    },
}

YP_OPS_TELEGRAM_NOTIFY = {
    'description': 'RUNTIMECLOUD-7626',
    'template_name': 'on_status_change',
    'template_kwargs': {
        'login': 'yp-monitoring-ops',
        'method': 'telegram',
        'status': [
            'CRIT',
        ],
    },
}


class ProductionLeafAggregator(aggregators.TimedMoreThanLimitIsProblem):
    def get_default_limits(self):
        return [
            {
                'crit': '1',  # exact 1 host may be lost before crit
                'day_end': 7,
                'day_start': 1,
                'time_end': 23,
                'time_start': 0,
                'warn': '0',
            },
        ]


class ProductionAggregator(rtc_aggregators.Production):
    leaf_aggregator = ProductionLeafAggregator


class ProductionLeafNotifications(rtc_notifications.EssentialNotifications):
    notify_rules = rtc_notifications.EssentialNotifications.notify_rules + (
        YP_OPS_EMAIL_NOTIFY,
        YP_OPS_TELEGRAM_NOTIFY,
    )


class ProductionNotifications(rtc_notifications.SelectiveNotifications):
    leaf_notifications = ProductionLeafNotifications
    meta_notifications = rtc_notifications.EssentialNotifications


class MutedLeafNotifications(rtc_notifications.EssentialNotifications):
    pass


class MutedNotifications(rtc_notifications.SelectiveNotifications):
    leaf_notifications = MutedLeafNotifications
    meta_notifications = rtc_notifications.EssentialNotifications


class OptFactory(RtcOptFactory):
    """
    Inject YP specific opt handlers.

    """
    def create_handler(self, handler_class, check):
        if issubclass(handler_class, rtc_aggregators.Production):
            return super().create_handler(ProductionAggregator, check)

        if issubclass(handler_class, rtc_notifications.Production):
            if check.__class__ in NOTIFABLE_CHECKS:
                return super().create_handler(ProductionNotifications, check)
            else:
                return super().create_handler(MutedNotifications, check)

        return super().create_handler(handler_class, check)


class Builder(JugglerChecksBuilder):
    def build_initial_tree(self, initial_data):
        return NodeSet(
            {
                'yp_masters': {
                    'children': {
                        'iva': {
                            'children': {'WALLE%PROD@prj=yp-iva': None},
                            'tags': ['grp_iva'],
                        },
                        'man': {
                            'children': {'WALLE%PROD@prj=yp-man': None},
                            'tags': ['grp_man'],
                        },
                        'man-pre': {
                            'children': {'WALLE%PROD@prj=yp-man-pre': None},
                            'tags': ['grp_man-pre'],
                        },
                        'msk': {
                            'children': {'WALLE%PROD@prj=yp-msk': None},
                            'tags': ['grp_msk'],
                        },
                        'sas': {
                            'children': {'WALLE%PROD@prj=yp-sas': None},
                            'tags': ['grp_sas'],
                        },
                        'sas-test': {
                            'children': {'WALLE%PROD@prj=yp-sas-test': None},
                            'tags': ['grp_sas-test'],
                        },
                        'vla': {
                            'children': {'WALLE%PROD@prj=yp-vla': None},
                            'tags': ['grp_vla'],
                        },
                        'xdc': {
                            'children': {'WALLE%PROD@prj=yp-xdc': None},
                            'tags': ['grp_xdc'],
                        },
                    },
                    'tags': ['prj_yp-masters'],
                },
            },
        )

    def build_checks_tree(self, initial_tree):
        return RtcCheckSetProd(
            initial_tree,
            opt_factory=OptFactory(),
            resolver=self.resolver,
        )


def main():
    Builder().run(app_desc=__doc__)
