from infra.reconf_juggler.builders import DeclaredChecksBuilder
from infra.reconf_juggler.opts import aggregators

from infra.rtc.juggler.reconf.checks.infrastructure import AbstractInfrastructureCheck
from infra.rtc.juggler.reconf.opts.notifications import EssentialNotifications


HOLD_CRIT_SECONDS = 970  # RUNTIMECLOUD-8893

NOTIFY_TELEGRAM_GROUPS = {
    'description': None,
    'template_name': 'on_status_change',
    'template_kwargs': {
        'login': [
            'Search_Runtime',
            'Yt_Runtime'
        ],
        'method': [
            'telegram',
        ],
        'status': [
            'CRIT',
            'OK'
        ]
    }
}


class YtPanelAggregatorAnd(aggregators.LogicAnd):
    def get_default_value(self, key):
        val = super().get_default_value(key)

        if key == 'aggregator_kwargs':
            if 'cpu_throt' in self._bound.name:
                if val is None:
                    val = {}
                val['hold_crit'] = HOLD_CRIT_SECONDS

        return val


class YtPanelAggregatorOr(aggregators.LogicOr):
    def get_default_value(self, key):
        val = super().get_default_value(key)

        if key == 'aggregator_kwargs':
            kwargs = {}

            if 'cpu_throt' in self._bound.name or 'performance_crit' in self._bound.name:
                kwargs['hold_crit'] = HOLD_CRIT_SECONDS

            if kwargs:
                if val is None:
                    val = kwargs
                else:
                    val.update(kwargs)

        return val


class YtPanelCheck(AbstractInfrastructureCheck):
    """FIXME: docs required (RUNTIMECLOUD-9527)"""
    doc_url = None  # should be defined in declarative builder

    _aggregator = YtPanelAggregatorOr
    _notifications = EssentialNotifications


class YtPanelCheckAnd(YtPanelCheck):
    """FIXME: docs required (RUNTIMECLOUD-9527)"""
    doc_url = None  # should be defined in declarative builder

    _aggregator = YtPanelAggregatorAnd


class YtPanelSubcheck(AbstractInfrastructureCheck):
    """FIXME: docs required (RUNTIMECLOUD-9527)"""
    doc_url = None  # should be defined in declarative builder

    _aggregator = YtPanelAggregatorOr


class YtPanelBuilder(DeclaredChecksBuilder):
    """
    Base class for YP panels builders (RUNTIMECLOUD-8893).

    """
    default_check_class = 'infra.rtc.juggler.reconf.builders.projects.yt.panels.YtPanelCheck'

    DEV_CLUSTERS = (
        'socrates',
        'hume'
    )

    def build_initial_tree(self, initial_data, names_map=None, **kwargs):
        """
        Desired checks structure. Will be populated by default class check if
        meta omitted in the node. When node meta defined it will be propagated
        to it's children. Empty dict for check body means existing Juggler
        aggregator, `None` stands for raw event.

        """
        checks = {
            '{prefix}{name}{suffix}:UNREACHABLE': {
                'children': {
                    '{prefix}{name}:UNREACHABLE': {}
                }
            },
            '{prefix}{name}{suffix}:cpu_throt_capping_check': {
                'children': {
                    '{prefix}{name}:cpu_throt_capping_check': {}
                }
            },
            '{prefix}{name}{suffix}:cpu_throt_frequency_check': {
                'children': {
                    '{prefix}{name}:cpu_throt_frequency_check': {}
                }
            },
            '{prefix}{name}{suffix}:cpu_throt_perf_check': {
                'children': {
                    '{prefix}{name}:cpu_throt_perf_check': {}
                }
            },
            '{prefix}{name}{suffix}:cpu_throt_thermal_check': {
                'children': {
                    '{prefix}{name}:cpu_throt_thermal_check': {}
                }
            },
            '{prefix}{name}{suffix}:performance': {
                'children': {
                    '{prefix}{name}:cpu_throt_capping_check': {},
                    '{prefix}{name}:cpu_throt_frequency_check': {},
                    '{prefix}{name}:cpu_throt_perf_check': {},
                    '{prefix}{name}:cpu_throt_thermal_check': {},
                    '{prefix}{name}:ssd_link': {},
                    '{prefix}{name}:fs_free_space_syst': {},
                    '{prefix}{name}:fs_free_space_serv': {},
                    '{prefix}{name}:fs_free_space_user': {},
                }
            }
        }
        odin_checks = {
            '{prefix}{name}{suffix}:lvc_and_rtc_problems': {
                'children': {
                    '{name}.yt.yandex.net:lost_vital_chunks': {},  # maintaned by YT team
                    '{prefix}{name}:UNREACHABLE': {}
                },
                'meta': {'reconf': {'class': 'infra.rtc.juggler.reconf.builders.projects.yt.panels.YtPanelCheckAnd'}}
            },
            '{prefix}{name}{suffix}:sort_result_and_performance': {
                'children': {
                    '{name}.yt.yandex.net:sort_result': {},
                    '{prefix}{name}{suffix}_subchecks:performance_crit': {
                        'children': {
                            '{prefix}{name}:cpu_throt_capping_check': {},
                            '{prefix}{name}:cpu_throt_frequency_check': {},
                            '{prefix}{name}:cpu_throt_perf_check': {},
                            '{prefix}{name}:cpu_throt_thermal_check': {},
                            '{prefix}{name}:ssd_link': {},
                            '{prefix}{name}:fs_free_space_syst': {},
                            '{prefix}{name}:fs_free_space_serv': {},
                            '{prefix}{name}:fs_free_space_user': {},
                        },
                        'meta': {'reconf': {'class': 'infra.rtc.juggler.reconf.builders.projects.yt.panels.YtPanelSubcheck'}}
                    }
                },
                'meta': {'reconf': {'class': 'infra.rtc.juggler.reconf.builders.projects.yt.panels.YtPanelCheckAnd'}}
            }
        }
        gpu_checks = {
            '{prefix}{name}{suffix}:gpu': {
                'children': {
                    '{prefix}{name}:gpumanager': {},
                    '{prefix}{name}:nvidia_containers': {},
                }
            }
        }

        if 'gpu' in names_map['name']:
            # clusters with GPU don't have their own odin.yt.yandex.net checks, but should have gpu-related checks
            panel = {**checks, **gpu_checks}
        else:
            panel = {**checks, **odin_checks}

        # build root panel check and add it to the set
        root_children = {}
        for check in panel:
            host, _ = check.split(':')
            if host == '{prefix}{name}{suffix}':
                root_children[check] = {}

        if names_map['name'] not in self.DEV_CLUSTERS:
            panel['yt_panel:{name}'.format(**names_map)] = {
                'children': root_children,
                'tags': [
                    'yt_panel_root',
                ],
            }

        return super().build_initial_tree(panel, names_map=names_map, **kwargs)
