import logging

from infra.reconf_juggler.builders import DeclaredChecksBuilder
from infra.reconf_juggler.opts import aggregators
from infra.rtc.juggler.reconf.checks import RtcCheck

from infra.rtc.juggler.reconf.checks.infrastructure import AbstractInfrastructureCheck


class RtcPanelAggregatorAnd(aggregators.LogicAnd):
    pass


class RtcPanelAggregatorOr(aggregators.LogicOr):
    pass


class RtcPanelCheck(AbstractInfrastructureCheck):
    """FIXME: docs required (RUNTIMECLOUD-9527)"""
    doc_url = None  # should be defined in declarative builder

    _aggregator = RtcPanelAggregatorOr


class RtcPanelCheckAnd(RtcPanelCheck):
    """FIXME: docs required (RUNTIMECLOUD-9527)"""
    doc_url = None  # should be defined in declarative builder

    _aggregator = RtcPanelAggregatorAnd


class RtcPanelClusterDefinition:
    """
    The class represents a cluster definition decoder (from panel definition dict)

    """

    def __init__(self, cluster_def):
        if isinstance(cluster_def, dict):
            if 'cluster_agg_name' not in cluster_def or not cluster_def['cluster_agg_name']:
                raise RuntimeError("Cluster definition dict must contain non-empty cluster_agg_name field")

            self._cluster_agg_name = cluster_def['cluster_agg_name']
            self._panel_agg_name = cluster_def.get('panel_agg_name', self._cluster_agg_name)

            if 'geo' in cluster_def and cluster_def['geo']:
                self._geo = cluster_def['geo'] if isinstance(cluster_def['geo'], tuple) else \
                    (cluster_def['geo'], ) if isinstance(cluster_def['geo'], str) else ()
        elif isinstance(cluster_def, str):
            self._cluster_agg_name = self._panel_agg_name = cluster_def
        else:
            raise RuntimeError("Cluster definition must be either str or dict")

    def __str__(self):
        return self._cluster_agg_name

    @property
    def cluster_agg_name(self):
        return self._cluster_agg_name

    @property
    def panel_agg_name(self):
        return self._panel_agg_name

    @property
    def geo(self):
        return self._geo

    _cluster_agg_name = ""
    _panel_agg_name = ""
    _geo = ()


class RtcPanelBuilder(DeclaredChecksBuilder):
    """
    Base class for panels builders (RUNTIMECLOUD-8893).

    """
    default_check_class = 'infra.rtc.juggler.reconf.builders.shared.panels.rtc.RtcPanelCheck'

    def build_initial_tree(self, initial_data, **kwargs):
        """
        Desired checks structure. Will be populated by default class check if
        meta omitted in the node. When node meta defined it will be propagated
        to it's children. Empty dict for check body means existing Juggler
        aggregator, `None` stands for raw event.-

        """

        if 'panel_definition' not in kwargs:
            raise RuntimeError("Panel definition is required")

        panel_definition = kwargs['panel_definition']

        if 'panel_users' not in panel_definition:
            raise RuntimeError("At least one user has to be defined")

        if 'clusters' not in panel_definition or not panel_definition['clusters']:
            raise RuntimeError("Non-empty clusters list is required in panel structure")

        panel = {}

        if 'routes' not in panel_definition or not panel_definition['routes']:
            raise RuntimeError("Non-empty routes field is required in panel definition")

        panel_definition_tags = panel_definition.get('tags', {})
        if 'cluster' not in panel_definition_tags:
            raise RuntimeError("Cluster tag name is required in panel tags definition")

        # Iterate users
        for panel_user in panel_definition['panel_users'].keys():
            user_settings = panel_definition['panel_users'][panel_user]

            user_settings_panel_suffix = user_settings.get('cluster_panel_suffix', '_{}'.format(panel_user))
            user_settings_tag_suffix = user_settings.get('tag_suffix', '_panel_{}'.format(panel_user))

            cluster_agg_name_tmpl = kwargs['names_map']['cluster_panel'] + user_settings_panel_suffix + ':{}'

            # Iterate clusters
            for cluster_def in panel_definition['clusters']:
                cluster = RtcPanelClusterDefinition(cluster_def)
                cluster_agg_name = cluster_agg_name_tmpl.format(cluster.panel_agg_name)
                cluster_agg_children = {}

                for route in panel_definition['routes']:
                    route_agg_name = '{}{}{}:{}'.format(cluster, kwargs['names_map']['route_panel_suffix'],
                                                        user_settings_panel_suffix, route['name'])
                    route_agg_children = {}

                    logging.debug("Filling metrics for route {}".format(route_agg_name))
                    self.fill_metrics(panel_user, cluster, route_agg_children, route['metrics'])

                    if route_agg_children:
                        if panel_definition['two_layers']:
                            for route_agg_child_name in route_agg_children:
                                route_agg_name_2nd_layer = '{}_{}'.format(
                                    route_agg_name,
                                    route_agg_child_name.split(":", 1)[1] if ':' in route_agg_child_name else route_agg_child_name)
                                panel[route_agg_name_2nd_layer] = {
                                    'children': {route_agg_child_name: route_agg_children[route_agg_child_name]},
                                    'tags': ['{}{}{}'.format(cluster, panel_definition_tags['route_suffix'],
                                                             user_settings_tag_suffix)]
                                }
                                if 'tags' in user_settings:
                                    panel[route_agg_name_2nd_layer]['tags'].extend(user_settings['tags'])
                                cluster_agg_children[route_agg_name_2nd_layer] = {}
                        else:
                            panel[route_agg_name] = {
                                'children': route_agg_children,
                                'tags': ['{}{}{}'.format(cluster, panel_definition_tags['route_suffix'],
                                                         user_settings_tag_suffix)]
                            }
                            if 'tags' in user_settings:
                                panel[route_agg_name]['tags'].extend(user_settings['tags'])

                            cluster_agg_children[route_agg_name] = {}
                # end of routes iteration

                panel[cluster_agg_name] = {
                    'children': cluster_agg_children,
                    'tags': ['{}{}'.format(panel_definition_tags['cluster'], user_settings_tag_suffix)]
                }
            # end of clusters iteration
        # end of users iteration
        return super(RtcPanelBuilder, self).build_initial_tree(panel, **kwargs)

    # TODO: refactor this function in order to return meta info about required metrics in a route
    #       instead of actual tree. This will let avoid additional aggregates just to make
    #       short names in the panels. Caller have to be refactored accordingly too.
    def fill_metrics(self, user, cluster, initial_metrics, metrics_definition):
        for metric in metrics_definition:
            if isinstance(metric, type) and issubclass(metric, RtcCheck):
                initial_metrics['{}:{}'.format(cluster, metric.provides())] = {}
            elif isinstance(metric, dict):
                # check whether the dict is correct
                if 'metrics' not in metric:
                    raise RuntimeError("Metric dict must contain metrics field")

                if set(metric.keys()).isdisjoint(['users', 'clusters', 'clusters-avoid'])\
                        and not ('type' in metric and metric['type'] in ('external', 'external-per-geo')):
                    raise RuntimeError("Selective metric dict must contain at least one selection field")

                if 'clusters' in metric and 'clusters-avoid' in metric:
                    raise RuntimeError("clusters and clusters-avoid fields are mutually exclusive")

                if (('users' in metric and user not in metric['users']) or
                        ('clusters' in metric and cluster.cluster_agg_name not in metric['clusters']) or
                        ('clusters-avoid' in metric and cluster.cluster_agg_name in metric['clusters-avoid'])):
                    logging.debug("Skipping metric fill for cluster {}".format(cluster))
                    continue

                logging.debug("Metric fill for cluster {}".format(cluster))

                if 'type' in metric:
                    logging.debug("Metric type: {}".format(metric['type']))
                    if metric['type'] == 'external':
                        for metric_cluster in metric['metrics']:
                            if cluster.cluster_agg_name == metric_cluster or metric_cluster == 'all':
                                for submetric in metric['metrics'][metric_cluster]:
                                    if isinstance(submetric, tuple):
                                        initial_metrics["{}_{}:{}".format(cluster, user, submetric[1])] = {
                                            'children': {submetric[0]: {}}
                                        }
                                    else:
                                        initial_metrics[submetric] = {}
                    elif metric['type'] == 'external-per-geo':
                        if not isinstance(metric['metrics'], tuple):
                            raise RuntimeError("Metrics field in per geo metric have to be a tuple")

                        geo_count = len(cluster.geo) if isinstance(cluster.geo, tuple) else 1
                        for cluster_geo in cluster.geo:
                            for metric_per_geo in metric['metrics']:
                                if isinstance(metric_per_geo, tuple):
                                    subchild = {'children': {metric_per_geo[0].format(geo=cluster_geo): {}}}
                                    if geo_count > 1:
                                        subchild_name = "{}_{}_{}:{}".format(cluster, cluster_geo, user,
                                                                             metric_per_geo[1].format(geo=cluster_geo))
                                    else:
                                        subchild_name = "{}_{}:{}".format(cluster, user,
                                                                          metric_per_geo[1].format(geo=cluster_geo))
                                    initial_metrics[subchild_name] = subchild

                                else:
                                    initial_metrics[metric_per_geo.format(geo=cluster_geo)] = {}
                    else:
                        raise RuntimeError("Unknown metrics type {}".format(metric['type']))
                else:
                    self.fill_metrics(user, cluster, initial_metrics, metric['metrics'])
                continue
            else:
                print(metric)
                raise RuntimeError("Metrics must be an instance of either RtcCheck or dict")
