import logging

from infra.reconf.util import symbols

from infra.rtc.juggler.reconf.checks import RTC_BUNDLE_HTTP_URL
from infra.rtc.juggler.reconf.checks.infrastructure import AbstractInfrastructureCheck
from infra.rtc.juggler.reconf.checks.unreachable import UNREACHABLE

from infra.rtc.juggler.reconf.opts import flaps, notifications, ttls


class AbstractHostmanCheck(AbstractInfrastructureCheck):
    """ Base class for all hostman checks based on push events. """
    doc_url = None  # meaningless for abstract check
    maintainers = ('nekto0n', 'vaspahomov', 'warwish')

    _flap = flaps.Disabled  # ttl is too high, HOSTMAN-376
    _notifications = notifications.Sre
    _ttl = ttls.HalfAnHour
    _unreach_service_classes = (UNREACHABLE,)


class AbstractHostmanUnitCheck(AbstractHostmanCheck):
    """ Base class for all hostman units checks. """
    category = 'hostman_unit'
    doc_url = None  # meaningless for abstract check

    def __init__(self, *args, **kwargs):
        self._unreach_service_classes = (UNREACHABLE, hostman)
        super().__init__(*args, **kwargs)


class MutedHostmanUnitCheck(AbstractHostmanUnitCheck):
    """ Base class for units checks with solomon notifications only. """
    doc_url = None  # meaningless for abstract check

    _notifications = notifications.EssentialNotifications


class hostman(AbstractInfrastructureCheck):
    """
    Ensure hostman has no issues with it's binary and was executed in past
    half an hour (HOSTMAN-361).

    This is real passive check executed by juggler agent (not a push event), so
    we can use standard ttl and flap detector, that's why
    AbstractInfrastructureCheck subclassed. Small TTL allows to use this
    check as dependency to all push-based hostman checks to avoid flood when
    hostman not executed by some reason.

    """
    doc_url = 'https://st.yandex-team.ru/HOSTMAN-361'
    maintainers = AbstractHostmanCheck.maintainers

    _notifications = AbstractHostmanCheck._notifications


class hostman_apt(AbstractHostmanCheck):
    """ Ensure apt works properly and packages may be installed on host. """
    doc_url = RTC_BUNDLE_HTTP_URL + '/checks/hostman_apt/README.md'

    @classmethod
    def provides(cls, is_node=True):
        # https://st.yandex-team.ru/RUNTIMECLOUD-13839
        if is_node:
            return super().provides(is_node=is_node)
        return 'hostman-apt'  # raw event


class hostman_salt(AbstractHostmanCheck):
    """ Ensure hostman applied salt configuration without errors. """
    doc_url = RTC_BUNDLE_HTTP_URL + '/checks/hostman_salt/README.md'

    @classmethod
    def provides(cls, is_node=True):
        return 'hostman-salt'


class system_service_yandex_hm_reporter(AbstractHostmanUnitCheck):
    """
    Check hostman reporter unit applied syccessfully.

    hm-reporter is non-homogenous (installed on subset of hosts), so can't be
    monitored by `hostman_unit_check_factory`, details in HOSTMAN-797

    """
    doc_url = 'https://a.yandex-team.ru/arc/trunk/arcadia/infra/hmserver/docs/HM_REPORTER.md'

    @classmethod
    def provides(cls, is_node=True):
        if is_node:
            return super().provides(is_node=is_node)

        return 'system-service-yandex-hm-reporter-ready'  # raw event


class system_service_yandex_solomon_sysmond(AbstractHostmanUnitCheck):
    """
    Check solomon agent unit applied syccessfully.

    solomon-sysmond is non-homogenous (installed on subset of hosts), so can't
    be monitored by `hostman_unit_check_factory`, details in HOSTMAN-797

    """
    doc_url = 'https://docs.yandex-team.ru/solomon/data-collection/sysmond/overview'

    @classmethod
    def provides(cls, is_node=True):
        if is_node:
            return super().provides(is_node=is_node)

        return 'system-service-yandex-solomon-sysmond-ready'  # raw event


def hostman_unit_check_factory(name, kind, opts):
    """ Per-unit checks classes generator. """
    event_prefix = {
        'PackageSet': 'package-set-',
        'PortoDaemon': 'porto-daemon-',
        'SystemService': 'system-service-',
        'TimerJob': 'timer-job-',
    }
    event_suffix = {
        'PackageSet': '-ready',
        'PortoDaemon': '-running',
        'SystemService': '-ready',
        'TimerJob': '-ready',
    }
    node_prefix = {
        'PackageSet': 'package_set_',
        'PortoDaemon': 'porto_daemon_',
        'SystemService': 'system_service_',
        'TimerJob': 'timer_job_'
    }

    base_class = MutedHostmanUnitCheck

    if 'base_class' in opts:
        try:
            base_class = symbols.SymLoader().load(opts['base_class'])
        except Exception:
            # leave base class unchanged to keep unit under monitoring (we
            # can't fail whole aggregates bunch build), just log error with
            # trace to ease debug a bit. Probably this case may be solved
            # properly in https://st.yandex-team.ru/HOSTMAN-801
            logging.error("Can't load: %s:", opts['base_class'], exc_info=True)

    if 'doc_url' in opts:
        doc_url = opts['doc_url']
    else:
        doc_url = 'https://st.yandex-team.ru/HOSTMAN-676'  # at least something

    @classmethod
    def provides(cls, is_node=True):
        if is_node:
            return node_prefix.get(kind, 'unsupported_hostman_unit_type_') + name

        return event_prefix.get(kind, '') + name + event_suffix.get(kind, '')

    return type(
        'hostman_unit_' + node_prefix.get(kind, '') + name,
        (base_class,),
        {
            'doc_url': doc_url,
            'provides': provides,
            'validate_class_docstring': False,
        }
    )
