import logging

from infra.reconf_juggler import CheckSet

from infra.rtc.juggler.reconf.checks import (
    hardware,
    hostman,
    infrastructure,
    informal,
    monitoring,
    cloud,
    network,
    unispace,
    unreachable,
    walle,
)


class CommonCheckSetBase(CheckSet):
    branches = (
        cloud.certman,
        cloud.check_iss_agent,
        cloud.check_skynet_procs,
        cloud.portod_tasks,
        cloud.skynet_copier,
        hardware.bad_raid_check,
        hardware.cpu_throt_capping_check,
        hardware.cpu_throt_check,
        hardware.cpu_throt_frequency_check,
        hardware.cpu_throt_perf_check,
        hardware.cpu_throt_thermal_check,
        hardware.cpu_throt_turboboost_check,
        hardware.link_utilization,
        hardware.walle_bmc,
        hardware.walle_cpu,
        hardware.walle_disk,
        hardware.walle_fs_check,
        hardware.walle_link,
        hardware.walle_memory,
        hardware.walle_tainted_kernel,
        hostman.hostman,
        hostman.hostman_apt,
        hostman.hostman_salt,
        informal.check_atop,
        informal.check_cpu,
        informal.cpu_state,
        informal.eth_errors,
        informal.fs_free_space_serv,
        informal.fs_free_space_user,
        informal.grub2_check,
        informal.io_monitor,
        informal.load_monitor,
        informal.need_reboot_kernel,
        informal.net_errors,
        informal.numa_memory,
        informal.raid_state,
        informal.skybone_locked_mem,
        informal.ssd_link,
        informal.walle_reboots,
        infrastructure.bind,
        infrastructure.check_dstate_count,
        infrastructure.cron,
        infrastructure.cron_errs,
        infrastructure.fs_mount_opts,
        infrastructure.ntp,
        infrastructure.packages_state,
        infrastructure.ssh,
        infrastructure.sysconf,
        monitoring.yasmagent_ping,
        network.abnormal_ip_conf,
        network.check_ra,
        network.fb_route_check,
        network.hbf_agent_rtc,
        network.ip_groups,
        network.ll_duplicate_addr,
        network.net_irq,
        network.tcp_sampler_dump,
        network.tsolro_check,
        unispace.fs_free_space_syst,
        unreachable.META,
        unreachable.UNREACHABLE,
        unreachable.bundle_info,
    )

    def get_initial_branches(self):
        # hostman branches are dynamic, so can't be simply put into a class
        # attribute (at least canonized resolver is not accessible when class
        # initialized).
        hostman_branches = []

        for filename, spec, opts, err in self.resolver['hostman']['units'].resolve(''):
            if err is not None:
                logging.error('Failed to load %s: %s', filename, err)
                continue

            try:
                # Strip trailing '@' as juggler does not allow it (hostctl does the same)
                unit_name = spec['meta']['name'].rstrip('@')
                unit_kind = spec['meta']['kind']
            except KeyError:
                logging.error('Failed to get name and type from: %s', filename)
                continue

            if not opts.get('generate_aggregates', True):
                logging.warning('Skip unit %s (%s: build disbaled)',
                                unit_name, filename)
                continue

            hostman_branches.append(hostman.hostman_unit_check_factory(
                unit_name, unit_kind, opts))

        return tuple(hostman_branches) + self.branches


class CommonCheckSetProd(CommonCheckSetBase):
    branches = CommonCheckSetBase.branches + (
        infrastructure.cauth_userd,
    )


class CommonCheckSetPrestable(CommonCheckSetBase):
    branches = CommonCheckSetBase.branches + (
        infrastructure.cauth_userd,
    )


class GpuCheckSet(CheckSet):
    branches = (
        hardware.gpumanager,
        infrastructure.nvidia_containers,
    )


class QloudCheckSet(CommonCheckSetBase):
    branches = CommonCheckSetBase.branches + (
        infrastructure.dns64_localcacher,
        infrastructure.qloud_logoped,
        infrastructure.qloud_logoped_statforward
    )


class RtcCheckSetProd(CommonCheckSetProd):
    branches = CommonCheckSetProd.branches + (
        infrastructure.rotmissingdocs,
    )


class RtcCheckSetPrestable(CommonCheckSetPrestable):
    branches = CommonCheckSetPrestable.branches + (
        infrastructure.rotmissingdocs,
    )


class WalleStatusCheckSet(CheckSet):
    branches = (
        walle.walle_status,
    )


class YtCheckSet(CommonCheckSetBase):
    branches = CommonCheckSetBase.branches + GpuCheckSet.branches + (
        hostman.system_service_yandex_solomon_sysmond,
    )
