"""
ReConf-Juggler-RTC monitoring coverage measuring tool (st/HOSTMAN-516)

Main idea is to substract one events set (target events, which coverage is
measured) from another set (base events, already emitted by desired hosts)

"""
import argparse
import json
import logging
import sys


from infra.reconf import cli
from infra.reconf_juggler import resolvers


log = logging.getLogger(__name__)


class CliTool(cli.CliApp):
    """ ReConf-Juggler-RTC monitoring coverage measuring tool """

    default_loglevel = 'INFO'

    def __init__(self, *args, resolver=None, **kwargs):
        super().__init__(*args, **kwargs)

        if resolver is None:
            self.resolver = resolvers.RootResolver()

    def get_argparser(self, *args, **kwargs):
        parser = super().get_argparser(*args, **kwargs)
        parser.add_argument(
            '--base-service-name',
            default='bundle_info',
            help='base service name; default is "%(default)s"',
            type=str,
        )
        parser.add_argument(
            '--base-service-tags',
            default=['HOSTMAN-516'],
            help='base service tags; default is %(default)s',
            type=self.split_arg_enum,
        )
        parser.add_argument(
            '--target-service-name',
            default='cron',
            help='measured service name; default is "%(default)s"',
            type=str,
        )
        parser.add_argument(
            '--target-service-tags',
            default=['HOSTMAN-516'],
            help='measured service tags; default is %(default)s',
            type=self.split_arg_enum,
        )
        parser.add_argument(
            '--group-by',
            choices=('NONE', 'WALLE-PROJECTS'),
            help='how to group uncovered hosts; "%(default)s" is default',
            default='NONE',
            type=str,
        )
        parser.add_argument(
            '--load-from-file',
            help='Load uncovered hosts from file (JSON list expected)',
            type=argparse.FileType(),
        )

        return parser

    def get_hosts_with_event(self, service_name, tags):
        # resolve_query() is intentionally used instead of resolve() to avoid
        # caching (events are changing over time)
        return set(
            x['host']
            for x in self.resolver['juggler']['raw_events'].resolve_query(
                {'filters': [{'service': service_name, 'tags': tags}]}
            )
        )

    def group_hosts(self, hosts):
        if self.args.group_by == 'WALLE-PROJECTS':
            hosts_info = self.resolver['walle']['hosts'].resolve({
                'names': hosts,
                'fields': ('name', 'project'),
            })

            groups = {}
            for i in hosts_info:
                groups.setdefault(i['project'], []).append(i['name'])

            for hosts_list in groups.values():
                hosts_list.sort()

        elif self.args.group_by == 'NONE':
            groups = sorted(hosts)

        return groups

    def run(self):
        if self.args.load_from_file is None:
            base_hosts = self.get_hosts_with_event(
                self.args.base_service_name,
                self.args.base_service_tags,
            )
            log.info('%i base hosts collected', len(base_hosts))

            covered_hosts = self.get_hosts_with_event(
                self.args.target_service_name,
                self.args.target_service_tags,
            )
            log.info('%i covered hosts collected', len(covered_hosts))

            uncovered = list(base_hosts.difference(covered_hosts))
        else:
            uncovered = json.load(self.args.load_from_file)

        log.info('%i uncovered hosts found', len(uncovered))

        if uncovered:
            log.debug('Uncovered hosts: %s', uncovered)
            uncovered = self.group_hosts(uncovered)

        json.dump(uncovered, sys.stdout, indent=3, sort_keys=True)


def main():
    CliTool().run()
