"""
Collection of RTC Juggler aggregators.

https://wiki.yandex-team.ru/sm/juggler/aggregation-modules

"""

from infra.reconf_juggler.opts import aggregators


class DescriptionMismatch(aggregators.SelectiveAggregator):
    """
    Return `logic_or` aggregator for metachecks and `description_mismatch` for
    leaf checks.

    """
    leaf_aggregator = aggregators.DescriptionMismatch


class InformalLeafAggregator(aggregators.LogicAnd):
    """
    Informal (informational) aggregator for leaf checks: crit when ALL
    children crits.

    """
    allow_empty_groups = True


class Informal(aggregators.SelectiveAggregator):
    """
    Return apropriate informal (informational) aggregator depending on
    tree level (no crits when at least one child alive).

    """
    leaf_aggregator = InformalLeafAggregator


class LogicOr(aggregators.LogicOr):
    """
    Crit when at least one child in crit state.

    """
    allow_empty_groups = True


class ProductionLeafAggregator(aggregators.TimedMoreThanLimitIsProblem):
    """
    RTC default production aggregator for leaf checks.

    """
    allow_empty_groups = True

    def get_default_limits(self):
        return [
            {
                'crit': '4%',
                'day_end': 5,
                'day_start': 1,
                'time_end': 9,
                'time_start': 0,
                'warn': '2%'
            },
            {
                'crit': '4%',
                'day_end': 7,
                'day_start': 6,
                'time_end': 23,
                'time_start': 0,
                'warn': '2%'
            },
            {
                'crit': '3%',
                'day_end': 5,
                'day_start': 1,
                'time_end': 23,
                'time_start': 9,
                'warn': '2%'
            },
        ]


class RelaxedLeafAggregator(ProductionLeafAggregator):
    """
    Relaxed production aggregator for leaf checks.

    """
    def get_default_limits(self):
        return [
            {
                'crit': '15%',
                'day_end': 7,
                'day_start': 1,
                'time_end': 23,
                'time_start': 0,
                'warn': '10%',
            },
        ]


class Production(aggregators.SelectiveAggregator):
    """
    Return apropriate production aggregator depending on tree level.

    """
    leaf_aggregator = ProductionLeafAggregator


class Relaxed(aggregators.SelectiveAggregator):
    """
    Return apropriate production aggregator depending on tree level.

    """
    leaf_aggregator = RelaxedLeafAggregator
