"""
Notifications for RTC Juggler checks.
For reference see https://wiki.yandex-team.ru/sm/juggler/notifications/

"""

from infra.reconf_juggler.opts.notifications import NotificationsHandler


SOLOMON_URL = 'https://solomon.yandex-team.ru/?project=juggler&cluster=checks&service=push&graph=auto'

TELEGRAM_HOSTMAN_DEV_CRITS = {
    'description': None,
    'template_name': 'on_status_change',
    'template_kwargs': {
        'login': 'hostman-dev',
        'method': 'telegram',
        'repeat': 14400,
        'status': [
            'CRIT',
        ],
    },
}

TELEGRAM_HOSTMAN_DEV_WARNS = {
    'description': None,
    'template_name': 'on_status_change',
    'template_kwargs': {
        'login': 'hostman-dev',
        'method': 'telegram',
        'status': [
            'WARN',
        ],
        'delay': 3600,
        'day_start': 1,
        'day_end': 5,
        'time_start': '11:00',
        'time_end': '19:00',
    },
}

TELEGRAM_NANNY_MONITORINGS = {
    'description': None,
    'template_name': 'on_status_change',
    'template_kwargs': {
        'login': 'nanny-monitorings',
        'method': 'telegram',
        'status': [
            'CRIT',
        ],
    },
}

TELEGRAM_SEARCH_RUNTIME = {
    'description': None,
    'template_name': 'on_status_change',
    'template_kwargs': {
        'login': 'Search_Runtime',
        'method': 'telegram',
        'repeat': 14400,
        'status': [
            'CRIT',
        ],
    },
}

TELEGRAM_RTC_INFRA = {
    'description': None,
    'template_name': 'on_status_change',
    'template_kwargs': {
        'login': 'Rtc_Infra',
        'method': 'telegram',
        'repeat': 14400,
        'status': [
            'CRIT',
        ],
    },
}

TELEGRAM_RTC_SYSDEV = {
    'description': None,
    'template_name': 'on_status_change',
    'template_kwargs': {
        'login': 'rtc-networking',
        'method': 'telegram',
        'repeat': 14400,
        'status': [
            'CRIT',
        ],
    },
}

TELEGRAM_HBF_TCP_SAMPLER_VALVE = {
    'description': None,
    'template_name': 'on_status_change',
    'template_kwargs': {
        'login': 'hbf-tcp-sampler-valve',
        'method': 'telegram',
        'repeat': 14400,
        'status': [
            'CRIT',
        ],
    },
}

TELEGRAM_SENSITIVE_WALLE = {
    'description': None,
    'template_name': 'on_status_change',
    'template_kwargs': {
        'login': 'wall-e_notifications',
        'method': 'telegram',
        'repeat': 7200,
        'status': [
            'CRIT',
        ],
    },
}

TELEGRAM_WALLE = {
    'description': None,
    'template_name': 'on_status_change',
    'template_kwargs': {
        'login': 'wall-e_notifications',
        'method': 'telegram',
        'repeat': 7200,
        'status': [
            'CRIT',
        ],
        'day_start': 1,
        'day_end': 5,
        'time_start': '08:00',
        'time_end': '19:00',
    },
}

SOLOMON = {
    'description': 'st/RUNTIMECLOUD-9489',
    'template_name': 'solomon',
    'template_kwargs': {},
}


class EssentialNotifications(NotificationsHandler):
    """
    Notifications every RTC aggregate should have

    """
    notify_rules = (
        SOLOMON,
    )

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

        # TODO(mixas): make Notifications and notify rules using handled
        # collections (opts are confs also) to be able to interact properly
        # and move this solomon stuff to reconf_juggler as generic feature
        if self._bound is not None:
            if self._bound.name == self._bound.origin:  # predefined (declared)
                jhost, jservice = self._bound.name.split(':')
            else:
                jhost = self._bound.origin
                jservice = self._bound.provides()

            uri = '&l.juggler_host={}&l.status=*&l.juggler_service={}'.format(
                jhost, jservice)
            meta_link = {
                'title': 'Chart',
                'url': SOLOMON_URL + uri,
                'type': 'solomon',
            }

            if meta_link not in self._bound['meta'].setdefault('urls', []):
                self._bound['meta']['urls'].append(meta_link)

    def get_handled_value(self, key, val):
        notifications = self.get_default_value(key)

        for item in val:
            if item not in notifications:
                notifications.append(item)

        return notifications


class ProductionLeafNotifications(EssentialNotifications):
    """
    RTC default production notifications for leaf checks.

    """
    notify_rules = EssentialNotifications.notify_rules + (
        TELEGRAM_SEARCH_RUNTIME,
    )


class HostmanLeafNotifications(EssentialNotifications):
    """
    Hostman team production notifications for leaf checks.

    """
    notify_rules = EssentialNotifications.notify_rules + (
        TELEGRAM_HOSTMAN_DEV_CRITS,
    )


class HostmanSensitiveLeafNotifications(HostmanLeafNotifications):
    notify_rules = HostmanLeafNotifications.notify_rules + (
        TELEGRAM_HOSTMAN_DEV_WARNS,
    )


class NannyLeafNotifications(EssentialNotifications):
    """
    Nanny team production notifications for leaf checks.

    """
    notify_rules = EssentialNotifications.notify_rules + (
        TELEGRAM_NANNY_MONITORINGS,
    )


class SysdevLeafNotifications(EssentialNotifications):
    """
    SYSDEV team production notifications for leaf checks.

    """
    notify_rules = EssentialNotifications.notify_rules + (
        TELEGRAM_RTC_SYSDEV,
    )


class SysdevNocLeafNotifications(EssentialNotifications):
    """
    SYSDEV + NOC team production notifications for leaf checks.

    """
    notify_rules = EssentialNotifications.notify_rules + (
        TELEGRAM_RTC_SYSDEV,
        TELEGRAM_HBF_TCP_SAMPLER_VALVE,
    )


class WalleLeafNotifications(EssentialNotifications):
    """
    Wall-E team production notifications for leaf checks.

    """
    notify_rules = EssentialNotifications.notify_rules + (
        TELEGRAM_WALLE,
    )


class WalleSensitiveLeafNotifications(EssentialNotifications):
    """
    Wall-E team production notifications for leaf checks.

    """
    notify_rules = EssentialNotifications.notify_rules + (
        TELEGRAM_SENSITIVE_WALLE,
    )


class TestingLeafNotifications(EssentialNotifications):
    """
    RTC testing notifications for leaf checks (RUNTIMECLOUD-8622).

    """
    notify_rules = EssentialNotifications.notify_rules + (
        TELEGRAM_RTC_INFRA,
    )


class SelectiveNotifications(NotificationsHandler):
    """
    Return apropriate notifications object depending on tree level.

    Almost always notifications are set to endpoint aggregators (aggregates
    raw events) - fine granularity.

    """
    leaf_notifications = EssentialNotifications
    meta_notifications = EssentialNotifications

    def __new__(cls, *args, bound=None, **kwargs):
        if not bound.has_subnodes() or bound.has_endpoints():
            notify_class = cls.leaf_notifications
        else:
            notify_class = cls.meta_notifications

        return notify_class(*args, bound=bound, **kwargs)


class Production(SelectiveNotifications):
    """
    Return apropriate production RTC notifications depending on tree level.

    """
    # Temporary disabled, RUNTIMECLOUD-13395
    # leaf_notifications = ProductionLeafNotifications


class Hostman(Production):
    """
    Hostman team production hotifications. Crits 24/7.

    """
    leaf_notifications = HostmanLeafNotifications


class HostmanSensitive(Hostman):
    """
    Hostman team production hotifications. Crits 24/7, warns in working hours.

    """
    leaf_notifications = HostmanSensitiveLeafNotifications


class Nanny(Production):
    """
    Nanny team production hotifications.

    """
    leaf_notifications = NannyLeafNotifications


class Sysdev(Production):
    """
    SYSDEV team production hotifications.

    """
    leaf_notifications = SysdevLeafNotifications


class SysdevNoc(Production):
    """
    SYSDEV + NOC teams production hotifications.

    """
    leaf_notifications = SysdevNocLeafNotifications


class Sre(Production):
    leaf_notifications = ProductionLeafNotifications


class WalleTg(Production):
    leaf_notifications = WalleLeafNotifications


class WalleSensitiveTg(Production):
    leaf_notifications = WalleSensitiveLeafNotifications


class Testing(SelectiveNotifications):
    """
    Return apropriate testing RTC notifications depending on tree level.

    """
    leaf_notifications = TestingLeafNotifications
