#!/usr/bin/env python3

from infra.rtc.jyggalag.jyggalag_thread import JyggalagThread
from infra.rtc.jyggalag.jyggalag_web import start as start_service
from infra.rtc.jyggalag.jyggalag_mind import JyggalagMind
from infra.rtc.jyggalag.jyggalag_config import JyggalagConfig
import argparse
import signal
import sys

default_jyggalag_fields = ['inv', 'name', 'jproject', 'jsegment', 'jstate', 'jok', 'approved', 'wproject', 'wstate',
                           'wstatus', 'qsegment', 'qstate', 'qslots']
default_list_fields = ['inv', 'name', 'wproject', 'qsegment', 'qstate', 'wstate', 'wstatus', 'qslots']
default_log_fields = ['time', 'inv', 'name', 'node', 'state_from', 'state_to']


def daemon_once(args):
    mind = JyggalagMind(create_config(args))
    mind.active = True
    mind.rotate()
    mind.active = False


def daemon(args):
    config = create_config(args)
    mind = JyggalagMind(config)
    signal.signal(signal.SIGTERM, mind.shutdown)
    signal.signal(signal.SIGINT, mind.shutdown)
    thread = JyggalagThread(mind)
    thread.start()
    start_service(config, mind)


def daemon_web(args):
    config = create_config(args)
    config.readonly = True
    mind = JyggalagMind(config)
    mind.lock_name = None
    signal.signal(signal.SIGTERM, mind.shutdown)
    signal.signal(signal.SIGINT, mind.shutdown)
    thread = JyggalagThread(mind)
    thread.start()
    start_service(config, mind)


def suspect(args):
    mind = JyggalagMind(create_config(args))
    hosts = get_data(args)
    resolved_hosts = mind.get_hosts(hosts)
    for host in resolved_hosts:
        if host.qloud is not None:
            mind.qloud.set_host_state(host.qloud, 'SUSPECT', args.message)
    print_jyggaloag_hosts(mind.get_hosts(hosts), default_list_fields)


def daemon_add(args):
    mind = JyggalagMind(create_config(args))
    mind.db.add_host(
        name=args.host,
        segment=args.segment,
        state=args.state
    )


def daemon_move(args):
    if args.segment is None:
        raise Exception('segment is not set')
    mind = JyggalagMind(create_config(args))
    hosts = get_data(args)
    actionable_hosts = list()
    for host in mind.get_hosts(hosts):
        if host.qloud is not None and host.qloud.segment != args.segment:
            actionable_hosts.append(host)

    for host in actionable_hosts:
        mind.db.add_host(
            name=host.name,
            segment=args.segment,
            state='QLOUD_MOVE_QUEUED'
        )


def daemon_list(args):
    mind = JyggalagMind(create_config(args))
    print_jyggaloag_hosts(mind.machine.get_hosts())
    pass


def daemon_timeout_list(args):
    mind = JyggalagMind(create_config(args))
    print_jyggaloag_hosts(mind.get_timeout_hosts())
    pass


def daemon_lastlog(args):
    mind = JyggalagMind(create_config(args))
    lastlog = mind.db.get_log(3600 * 24 * 30)
    print_jyggalag_log(lastlog, default_log_fields)
    pass


def daemon_ok(args):
    mind = JyggalagMind(create_config(args))
    for host in get_data(args):
        mind.db.ok_host(name=host)


def daemon_ok_all(args):
    mind = JyggalagMind(create_config(args))
    mind.db.ok_all()


def daemon_pending_list(args):
    mind = JyggalagMind(create_config(args))
    print_jyggaloag_hosts(mind.get_penging_hosts())


def daemon_remove(args):
    mind = JyggalagMind(create_config(args))
    for host in get_data(args):
        mind.db.remove_host(name=host)


def down(args):
    mind = JyggalagMind(create_config(args))
    hosts = get_data(args)
    resolved_hosts = mind.get_hosts(hosts)
    for host in resolved_hosts:
        if host.qloud is not None:
            mind.qloud.set_host_state(host.qloud, 'DOWN', args.message)
    print_jyggaloag_hosts(mind.get_hosts(hosts), default_list_fields)
    pass


def segment_list(args):
    if args.segment is None:
        return
    mind = JyggalagMind(create_config(args))
    qloud_hosts = mind.qloud.get_segment(segment=args.segment)
    names = list()
    for host in qloud_hosts:
        names.append(host.name)
    print_jyggaloag_hosts(mind.get_hosts(names), default_list_fields)


def hosts_list(args):
    mind = JyggalagMind(create_config(args))
    print_jyggaloag_hosts(mind.get_hosts(get_data(args)), default_list_fields)


def up(args):
    mind = JyggalagMind(create_config(args))
    hosts = get_data(args)
    resolved_hosts = mind.get_hosts(hosts)
    for host in resolved_hosts:
        if host.qloud is not None:
            mind.qloud.set_host_state(host.qloud, 'UP', args.message)
    print_jyggaloag_hosts(mind.get_hosts(hosts), default_list_fields)


def host_state(args):
    if args.state is None:
        return
    mind = JyggalagMind(create_config(args))
    hosts = get_data(args)
    resolved_hosts = mind.get_hosts(hosts)
    for host in resolved_hosts:
        if host.qloud is not None:
            mind.qloud.set_host_state(host.qloud, args.state, args.message)
    print_jyggaloag_hosts(mind.get_hosts(hosts), default_list_fields)
    pass


def add(args):
    if args.segment is None:
        return
    mind = JyggalagMind(create_config(args))
    hosts = get_data(args)
    resolved_hosts = mind.get_hosts(hosts)
    for host in resolved_hosts:
        mind.qloud.add_host(host.name, args.segment)
    print_jyggaloag_hosts(mind.get_hosts(hosts), default_list_fields)


def services(args):
    mind = JyggalagMind(create_config(args))
    hosts = get_data(args)
    service_result = dict()
    for host in mind.get_hosts(hosts):
        if host.qloud is not None:
            for slot in host.qloud.services.keys():
                service = host.qloud.services[slot]
                name = service.get_full_env_name()
                if name not in service_result:
                    service_result[name] = 0
                service_result[name] += 1
    for service in service_result.keys():
        print('{count} {name}'.format(count=service_result[service], name=service))


def switch_segment(args):
    if args.segment is None:
        return
    mind = JyggalagMind(create_config(args))
    hosts = get_data(args)
    resolved_hosts = mind.get_hosts(hosts)
    for host in resolved_hosts:
        if host.qloud is not None:
            mind.qloud.change_host_segment(host.qloud, args.segment, args.message)
    print_jyggaloag_hosts(mind.get_hosts(hosts), default_list_fields)


def meta(args):
    mind = JyggalagMind(create_config(args))
    hosts = get_data(args)
    resolved_hosts = mind.get_hosts(hosts)
    request = list()
    for host in resolved_hosts:
        if host.qloud is not None:
            request.append(host.qloud)
    mind.qloud.update_metas(request)
    print_jyggaloag_hosts(mind.get_hosts(hosts), default_list_fields)


def remove(args):
    mind = JyggalagMind(create_config(args))
    hosts = get_data(args)
    for host in mind.get_hosts(hosts):
        if host.qloud is not None:
            mind.qloud.remove_host(host.qloud)
    print_jyggaloag_hosts(mind.get_hosts(hosts), default_list_fields)


def get_data(args):
    data = list()
    if args.host is None:
        if args.file is None:
            for line in sys.stdin.readlines():
                data.append(str(line).strip('\n'))
        else:
            with open(args.file) as f:
                for line in f.readlines():
                    data.append(str(line).strip('\n'))
    else:
        data = list()
        data.append(args.host)
    return data


def print_jyggalag_log(log, fields=None):
    if fields is None:
        fields = default_jyggalag_fields
    spaces = list()
    for field in fields:
        spaces.append(len(field))
    results = list()
    for host in log:
        log_fields = host.str_columns(fields)
        results.append(log_fields)
        field_index = 0
        for host_field in log_fields:
            current_size = len(str(host_field))
            if spaces[field_index] < current_size:
                spaces[field_index] = current_size
            field_index += 1

    field_index = 0
    line = ''
    for field_name in fields:
        line += field_name.ljust(spaces[field_index]) + ' '
        field_index += 1
    print(line)

    for log_fields in results:
        field_index = 0
        line = ''
        for log_field in log_fields:
            line += str(log_field).ljust(spaces[field_index]) + ' '
            field_index += 1
        print(line)


def print_jyggaloag_hosts(hosts, fields=None):
    if fields is None:
        fields = default_jyggalag_fields
    spaces = list()
    for field in fields:
        spaces.append(len(field))
    results = list()
    for host in hosts:
        host_fields = host.str_columns(fields)
        results.append(host_fields)
        field_index = 0
        for host_field in host_fields:
            current_size = len(str(host_field))
            if spaces[field_index] < current_size:
                spaces[field_index] = current_size
            field_index += 1

    field_index = 0
    line = ''
    for field_name in fields:
        line += field_name.ljust(spaces[field_index]) + ' '
        field_index += 1
    print(line)

    for host_fields in results:
        field_index = 0
        line = ''
        for host_field in host_fields:
            line += str(host_field).ljust(spaces[field_index]) + ' '
            field_index += 1
        print(line)


def create_config(args):
    if args.config is None:
        return JyggalagConfig()
    else:
        return JyggalagConfig(args.config)


if __name__ == '__main__':
    argparser = argparse.ArgumentParser(description='Jyggalag')
    argparser.add_argument('--config', '-c', nargs='?', type=str, dest='config', action='store', help='Config')
    subparsers = argparser.add_subparsers(
        title='subcommands',
        help='additional help'
    )

    add_host_parser = subparsers.add_parser('hosts-add', description='Add hosts to qloud')
    add_host_parser.add_argument('--host', nargs='?', type=str, dest='host', action='store', help='Host')
    add_host_parser.add_argument('--file', nargs='?', type=str, dest='file', action='store', help='Hosts File')
    add_host_parser.add_argument('segment', action='store', type=str, help='Segment')
    add_host_parser.set_defaults(func=add)

    remove_host_parser = subparsers.add_parser('hosts-remove', description='Remove hosts from qloud')
    remove_host_parser.add_argument('--host', nargs='?', dest='host', action='store', type=str, help='Host')
    remove_host_parser.add_argument('--file', nargs='?', type=str, dest='file', action='store', help='Hosts File')
    remove_host_parser.set_defaults(func=remove)

    list_parser = subparsers.add_parser('segment-list', description='List qloud segment')
    list_parser.add_argument('segment', action='store', type=str, help='Qloud Segment')
    list_parser.set_defaults(func=segment_list)

    list_segment_parser = subparsers.add_parser('hosts-list')
    list_segment_parser.add_argument('--host', nargs='?', dest='host', action='store', type=str, help='Host')
    list_segment_parser.add_argument('--file', nargs='?', type=str, dest='file', action='store', help='Hosts File')
    list_segment_parser.set_defaults(func=hosts_list)

    update_meta_parser = subparsers.add_parser('hosts-meta', description='Force update meta on hosts')
    update_meta_parser.add_argument('--host', nargs='?', dest='host', action='store', type=str, help='Host')
    update_meta_parser.add_argument('--file', nargs='?', type=str, dest='file', action='store', help='Hosts File')
    update_meta_parser.set_defaults(func=meta)

    change_segment_parser = subparsers.add_parser('hosts-switch-segment')
    change_segment_parser.add_argument('--host', nargs='?', dest='host', action='store', type=str, help='Host')
    change_segment_parser.add_argument('--file', nargs='?', type=str, dest='file', action='store', help='Hosts File')
    change_segment_parser.add_argument('segment', action='store', type=str, help='Segment')
    change_segment_parser.add_argument('--message', '-m', nargs='?', type=str, default='Jyggalag Manual State Change',
                                       dest='message', action='store', help='Message')
    change_segment_parser.set_defaults(func=switch_segment)

    up_parser = subparsers.add_parser('hosts-up')
    up_parser.add_argument('--host', nargs='?', dest='host', action='store', type=str, help='Host')
    up_parser.add_argument('--file', nargs='?', type=str, dest='file', action='store', help='Hosts File')
    up_parser.add_argument('--message', '-m', nargs='?', type=str, default='Jyggalag Manual State Change',
                           dest='message', action='store', help='Message')
    up_parser.set_defaults(func=up)

    state_parser = subparsers.add_parser('hosts-state')
    state_parser.add_argument('--host', nargs='?', dest='host', action='store', type=str, help='Host')
    state_parser.add_argument('--file', nargs='?', type=str, dest='file', action='store', help='Hosts File')
    state_parser.add_argument('--message', '-m', nargs='?', type=str, default='Jyggalag Manual State Change',
                              dest='message', action='store', help='Message')
    state_parser.add_argument('state', action='store', type=str, help='HFSM State')
    state_parser.set_defaults(func=host_state)

    down_parser = subparsers.add_parser('hosts-down', aliases=['hd'])
    down_parser.add_argument('--host', nargs='?', dest='host', action='store', type=str, help='Host')
    down_parser.add_argument('--file', nargs='?', type=str, dest='file', action='store', help='Hosts File')
    down_parser.add_argument('--message', '-m', nargs='?', type=str, default='Jyggalag Manual State Change',
                             dest='message', action='store', help='Message')
    down_parser.set_defaults(func=down)

    suspect_parser = subparsers.add_parser('hosts-suspect', aliases=['hsus'])
    suspect_parser.add_argument('--host', nargs='?', dest='host', action='store', type=str, help='Host')
    suspect_parser.add_argument('--file', nargs='?', type=str, dest='file', action='store', help='Hosts File')
    suspect_parser.add_argument('--message', '-m', nargs='?', type=str, default='Jyggalag Manual State Change',
                                dest='message', action='store', help='Message')
    suspect_parser.set_defaults(func=suspect)

    services_parser = subparsers.add_parser('hosts-services')
    services_parser.add_argument('--host', nargs='?', dest='host', action='store', type=str, help='Host')
    services_parser.add_argument('--file', nargs='?', type=str, dest='file', action='store', help='Hosts File')
    services_parser.set_defaults(func=services)

    daemon_parser = subparsers.add_parser('daemon', aliases=['d'])
    daemon_parser.set_defaults(func=daemon)

    daemon_web_parser = subparsers.add_parser('daemon-web', aliases=['dw'])
    daemon_web_parser.set_defaults(func=daemon_web)

    daemon_parser = subparsers.add_parser('daemon-once', aliases=['do'])
    daemon_parser.set_defaults(func=daemon_once)

    daemon_list_parser = subparsers.add_parser('daemon-list', aliases=['dl'])
    daemon_list_parser.set_defaults(func=daemon_list)

    daemon_ok_parser = subparsers.add_parser('daemon-ok')
    daemon_ok_parser.add_argument('--host', nargs='?', dest='host', action='store', type=str, help='Host')
    daemon_ok_parser.add_argument('--file', nargs='?', type=str, dest='file', action='store', help='Hosts File')
    daemon_ok_parser.set_defaults(func=daemon_ok)

    daemon_ok_all_parser = subparsers.add_parser('daemon-ok-all')
    daemon_ok_all_parser.set_defaults(func=daemon_ok_all)

    daemon_pending_list_parser = subparsers.add_parser('daemon-pending')
    daemon_pending_list_parser.set_defaults(func=daemon_pending_list)

    daemon_timeout_list_parser = subparsers.add_parser('daemon-timeouts')
    daemon_timeout_list_parser.set_defaults(func=daemon_timeout_list)

    daemon_lastlog_parser = subparsers.add_parser('daemon-lastlog')
    daemon_lastlog_parser.set_defaults(func=daemon_lastlog)

    daemon_remove_parser = subparsers.add_parser('daemon-remove')
    daemon_remove_parser.add_argument('--host', nargs='?', dest='host', action='store', type=str, help='Host')
    daemon_remove_parser.add_argument('--file', nargs='?', type=str, dest='file', action='store', help='Hosts File')
    daemon_remove_parser.set_defaults(func=daemon_remove)

    add_host_daemon = subparsers.add_parser('daemon-add')
    add_host_daemon.add_argument('--host', dest='host', action='store', type=str, help='Host')
    add_host_daemon.add_argument('segment', action='store', type=str, help='Segment (qloud-ext.common)')
    add_host_daemon.add_argument('--state', dest='state', action='store', type=str,
                                 help='State')
    add_host_daemon.set_defaults(func=daemon_add)

    add_move_host_daemon = subparsers.add_parser('daemon-move')
    add_move_host_daemon.add_argument('--host', dest='host', action='store', type=str, help='Host')
    add_move_host_daemon.add_argument('--segment', dest='segment', action='store', type=str,
                                      help='Segment (qloud-ext.common)')
    add_move_host_daemon.add_argument('--file', nargs='?', type=str, dest='file', action='store', help='Hosts File')
    add_move_host_daemon.set_defaults(func=daemon_move)

    args = argparser.parse_args()
    args.func(args)
