from infra.rtc.jyggalag.clients.qloud_service import QloudService
from infra.rtc.jyggalag.clients.qloud_host import QloudHost
from infra.rtc.jyggalag.jyggalag_config import JyggalagConfig
from requests.exceptions import HTTPError
import datetime
import requests
from urllib.parse import quote_plus
import time
import json
import sys

dc_aliases = {
    'MOSCOW': {'IVA', 'MYT'},
    'MSK': {'IVA', 'MYT'}
}

datacenters = {'SAS', 'MAN', 'VLA', 'MYT', 'IVA'}

down_states = {'DOWN', 'MAINTENANCE', 'PREPARE_MAINTENANCE'}
up_states = {'UP'}


class QloudApi:

    def __init__(self, config: JyggalagConfig = None):
        self.api_host = config.qloud['api_host']
        self.connect_timeout = float(config.qloud.get('connect_timeout', 0.5))
        self.read_timeout = float(config.qloud.get('read_timeout', 30))
        self.token = config.qloud.get('oauth_token', None)
        self.segment_project_map = config.qloud.get('segment_project_map', None)
        self.logger = config.logger
        self.fatal_hosts_diff = int(config.qloud.get('fatal_hosts_diff', 50))
        self.ignore_up_states = set(config.qloud.get('ignore_up_states', ['MAINTENANCE', 'PREPARE_MAINTENANCE']))
        self.search_protector = dict()
        for installation in self.api_host.keys():
            self.search_protector[installation] = 0

    def prep_request(self,
                     method: str,
                     installation: str,
                     url: str,
                     data=None,
                     headers: dict = None,
                     args: dict = None):
        if headers is None:
            headers = dict()
        if args is not None and len(args) > 0:
            argvalues = list()
            for key in args.keys():
                argvalues.append('{key}={value}'.format(key=quote_plus(key), value=quote_plus(args[key])))
            url = "{url}?{args}".format(url=url, args='&'.join(argvalues))
        headers['Authorization'] = 'OAuth {auth}'.format(auth=self.token)
        req = requests.request(
            method=method,
            url='{host}{url}'.format(host=self.api_host[installation], url=url),
            verify=False,
            data=data,
            headers=headers,
            timeout=(self.connect_timeout, self.read_timeout)
        )
        return req

    def update_metas(self, hosts: list):
        data = dict()
        for host in hosts:
            if host.installation not in data:
                data[host.installation] = list()
            data[host.installation].append(host.name)
        for installation in data.keys():
            req = self.prep_request(
                method='POST',
                installation=installation,
                headers={'Content-Type': 'application/json'},
                url='/api/v1/admin/hostupdate',
                data=json.dumps(data[installation])
            )
            req.raise_for_status()

    def get_json(self, installation: str, url: str):
        num = 3
        last_ex = None
        while num > 0:
            try:
                req = self.prep_request(
                    method='GET',
                    installation=installation,
                    url=url)
                req.raise_for_status()
                return req.json()
            except HTTPError as http_ex:
                if http_ex.response.status_code == 404 or http_ex.response.status_code == 403:
                    return None
                else:
                    raise http_ex
            except Exception as ex:
                last_ex = ex
                num -= 1
                time.sleep(0.5)
        raise last_ex

    def remove_host(self, host: QloudHost):
        if host is None:
            return
        req = self.prep_request(
            method='DELETE',
            installation=host.installation,
            url='/api/v1/hosts/{host}'.format(host=host.name)
        )
        req.raise_for_status()

    def add_host(self, hostname: str, segment: str):
        installation, segment, dc = self.parse_segment(segment)
        req = self.prep_request(
            method='POST',
            installation=installation,
            url='/api/v1/hosts',
            args={'boxFqdn': hostname, 'hardwareSegment': segment}
        )
        req.raise_for_status()

    def change_host_segment(self, host: QloudHost, segment: str, comment: str = str(datetime.datetime.now())):
        installation, segment_name, dc = self.parse_segment(segment)
        if host.installation != installation:
            self.logger.info('Host {} not in target installation'.format(host.name))
            self.logger.info('Removing {} from {}'.format(host.name, host.segment))
            self.remove_host(host)
            self.logger.info('Adding {} to {}'.format(host.name, segment))
            self.add_host(host.name, segment)
            host.installation = installation
            host.segment = segment
            return
        req = self.prep_request(
            method='PUT',
            installation=installation,
            url='/api/v1/hosts/{name}'.format(name=host.name),
            args={'hardwareSegment': segment_name, 'comment': comment}
        )
        req.raise_for_status()
        req.close()

    def set_host_state(self, host: QloudHost, state: str, comment: str = str(datetime.datetime.now())):
        req = self.prep_request(
            method='PUT',
            installation=host.installation,
            url='/api/v1/hosts/{name}'.format(name=host.name),
            args={'state': state, 'comment': comment}
        )
        req.raise_for_status()
        req.close()

    def get_stable_env_version(self, installation: str, env_name: str):
        req_url = '/api/v1/environment/stable/{env}'.format(env=env_name)
        data = self.get_json(installation, req_url)
        return str(data['version'])

    def parse_segment(self, segment: str):
        splitted = segment.split('.')
        installation = splitted[0].lower()
        installation_segment = None
        segment_dc = None

        if len(splitted) == 1:
            if installation not in self.api_host:
                installation = None
            pass
        elif len(splitted) == 2:
            installation_segment = splitted[1]
            pass
        elif len(splitted) == 3:
            installation_segment = splitted[1]
            segment_dc = splitted[2].upper()
            pass
        return installation, installation_segment, segment_dc

    def find_hosts(self, hosts_names: list, resolve_services: bool = False):
        result = dict()
        names_set = set(hosts_names)
        services_to_resolve = set()
        services_data = dict()

        diff_check = dict()
        for installation in self.api_host.keys():
            diff_check[installation] = 0

        for host in self.get_segment(""):
            diff_check[host.installation] += 1
            if host.name in names_set:
                for slot in host.services.keys():
                    services_to_resolve.add(host.services[slot].get_full_name())
                result[host.name] = host
        result_list = list()

        for installation in self.api_host.keys():
            if diff_check[installation] == 0:
                raise Exception("FATAL HOSTS COUNT {}".format(json.dumps(diff_check)))
            if self.search_protector[installation] - diff_check[installation] > self.fatal_hosts_diff:
                raise Exception("FATAL HOSTS DIFF {}".format(json.dumps(diff_check)))
        self.search_protector = diff_check

        for name in result.keys():
            result_list.append(result[name])

        if resolve_services:
            for service in services_to_resolve:
                services_data[service] = self.get_service_data(service)
            for host in result_list:
                for slot in host.services.keys():
                    slot_service_data = services_data[host.services[slot].get_full_name()]
                    if slot_service_data is not None:
                        self.fill_service_data(host.services[slot], slot_service_data)
                    else:
                        sys.stderr.write('Service {slot} was not found on {name}\n'.format(
                            slot=slot.get_full_name(),
                            name=host.name
                        ))
        return result_list

    def get_segment_walle_project(self, segment: str):
        if self.segment_project_map is None:
            raise Exception('segment_project_map is not set')
        if segment in self.segment_project_map:
            return self.segment_project_map[segment]
        return self.segment_project_map[segment.split('.')[0]]

    def get_segment(self, segment: str, resolve_services: bool = False):
        inst, seg, dc = self.parse_segment(segment)
        result = list()
        req_url = '/api/v1/hosts/search'
        if seg is not None:
            req_url += '?hardwareSegment={seg}'.format(seg=seg)
        if dc is not None:
            if dc in dc_aliases:
                for alias in dc_aliases[dc]:
                    result.extend(self.get_segment('{inst}.{seg}.{dc}'.format(
                        inst=inst,
                        seg=seg,
                        dc=alias
                    )))
                return result
            req_url += '&datacenter={dc}'.format(dc=dc)
        if inst is None:
            for inst in self.api_host.keys():
                for host_data in self.get_json(inst, req_url):
                    try:
                        host = QloudHost(installation=inst, data=host_data)
                        result.append(host)
                    except:
                        self.logger.exception('error on get segment host {}'.format(host_data))
        else:
            for host_data in self.get_json(inst, req_url):
                host = QloudHost(installation=inst, data=host_data)
                result.append(host)

        result_dict = dict()
        names = list()
        for host in result:
            result_dict[host.name] = host
            names.append(host.name)

        if resolve_services:
            services_to_resolve = set()
            for h in result:
                for slot in h.services.keys():
                    services_to_resolve.add(h.services[slot].get_full_name())

            services_data = dict()
            for service in services_to_resolve:
                services_data[service] = self.get_service_data(service)

            for host in result:
                for slot in host.services.keys():
                    slot_service_data = services_data[host.services[slot].get_full_name()]
                    if slot_service_data is not None:
                        self.fill_service_data(host.services[slot], slot_service_data)
                    else:
                        sys.stderr.write('Service {slot} was not found on {name}\n'.format(
                            slot=slot.get_full_name(),
                            name=host.name
                        ))
        return result

    def get_service_env(self, service: QloudService):
        env_name = '.'.join(service.service.split('.')[0:3])
        return service.installation, env_name

    def get_service_data(self, fullname: str):
        splitted = fullname.split('.')
        env_name = '.'.join(splitted[1:4])
        component = '.'.join(splitted[1:5])
        installation = splitted[0]
        version = self.get_stable_env_version(installation, env_name)
        req_url = '/api/v1/runtime/{component}/{version}'.format(component=component, version=version)
        return self.get_json(installation, req_url)

    def fill_service_data(self, service: QloudService, data: dict):
        if data is None:
            print('Service: {service} not found'.format(service=service.get_full_name()))
            return
        service.segment = str(data['hardwareSegment'])
        size_list = str(data['size']).split(';')
        service.cpu = float(size_list[1])
        service.memory = float(size_list[0])
        service.disk = float(data['diskSize'])
        service.disk += float(data['ediskSize'])
        for loc in data['instanceGroups']:
            dc_found = False
            for dc in datacenters:
                if str(loc['location']).upper().startswith(dc):
                    service.units[dc] = int(loc['units'])
                    dc_found = True
                    break
            if not dc_found:
                for vdc in dc_aliases.keys():
                    if str(loc['location']).upper().startswith(vdc):
                        service.units[vdc] = int(loc['units'])
                        dc_found = True
                        break
            if not dc_found:
                raise Exception('Unknown Location {loc}'.format(loc=str(loc["location"])))

    def get_service_link(self, service_full_name: str):
        splitted = service_full_name.split('.')
        installation = splitted[0]
        return '{}/projects/{}'.format(self.api_host[installation], '/'.join(splitted[1:4]))

    def fill_service(self, service: QloudService):
        env_name = '.'.join(service.service.split('.')[0:3])
        version = self.get_stable_env_version(service.installation, env_name)
        req_url = '/api/v1/runtime/{service}/{version}'.format(service=service.service, version=version)
        data = self.get_json(service.installation, req_url)
        service.segment = str(data['hardwareSegment'])
        size_list = str(data['size']).split(';')
        service.cpu = float(size_list[1])
        service.memory = float(size_list[0])
        service.disk = float(data['diskSize'])
        service.disk += float(data['ediskSize'])
        for loc in data['instanceGroups']:
            dc_found = False
            for dc in datacenters:
                if str(loc['location']).upper().startswith(dc):
                    service.units[dc] = int(loc['units'])
                    dc_found = True
                    break
            if not dc_found:
                for vdc in dc_aliases.keys():
                    if str(loc['location']).upper().startswith(vdc):
                        service.units[vdc] = int(loc['units'])
                        dc_found = True
                        break
            if not dc_found:
                raise Exception('Unknown Location {loc}'.format(loc=str(loc["location"])))
