from infra.rtc.jyggalag.clients.resolver_instance import ResolverInstance
from infra.rtc.jyggalag.clients.resolver_host import ResolverHost
from infra.rtc.jyggalag.jyggalag_config import JyggalagConfig
from infra.rtc.jyggalag.clients.qloud_tvm import QloudTVM
from requests.exceptions import HTTPError
from urllib.parse import quote_plus
import requests
import time


class ServiceResolver:

    def __init__(self, config: JyggalagConfig):
        self.tvm_destination = config.resolver['tvm_destination']
        self.api_host = config.resolver.get('api_host', 'https://rtc-instance-resolver.in.yandex-team.ru')
        self.connect_timeout = float(config.resolver.get('connect_timeout', 0.5))
        self.read_timeout = float(config.resolver.get('read_timeout', 10))
        self.tvm = QloudTVM(config)
        self.ignore_services_filter = dict()
        if 'ignore_services_filter' in config.resolver:
            for filter_name in config.resolver['ignore_services_filter'].keys():
                self.ignore_services_filter[filter_name] = set(config.resolver['ignore_services_filter'][filter_name])

    def prep_request(self, method: str, url: str, data=None, headers: dict = None, args: dict = None):
        if headers is None:
            headers = dict()

        if args is not None and len(args) > 0:
            argvalues = list()
            for key in args.keys():
                argvalues.append('{key}={value}'.format(key=quote_plus(key), value=quote_plus(args[key])))
            url = "{url}?{args}".format(url=url, args='&'.join(argvalues))
        try:
            headers['X-Ya-Service-Ticket'] = self.tvm.get_ticket(self.tvm_destination)
        except Exception as ex:
            raise Exception("Failed to obtain TVM ticket for rtc-resolver {0}".format(str(ex)))
        req = requests.request(
            method=method,
            url='{host}{url}'.format(host=self.api_host, url=url),
            verify=False,
            data=data,
            headers=headers
        )
        return req

    def get_json(self, url: str):
        num = 3
        last_ex = None
        while num > 0:
            try:
                req = self.prep_request(
                    method='GET',
                    url=url,
                    headers={'Accept': 'application/json'}
                )
                req.raise_for_status()
                return req.json()
            except HTTPError as http_ex:
                if http_ex.response.status_code == 404 or http_ex.response.status_code == 403:
                    return None
                else:
                    raise http_ex
            except Exception as ex:
                last_ex = ex
                num -= 1
                time.sleep(0.5)
        raise last_ex

    def parse_resolver_host(self, data: dict) -> ResolverHost:
        resolved_host = ResolverHost(data)
        if 'Instances' in data:
            for instance in data['Instances']:
                ignore = False
                for instance_field in instance.keys():
                    if (instance_field in self.ignore_services_filter and
                            instance[instance_field] in self.ignore_services_filter[instance_field]):
                        ignore = True
                if ignore:
                    continue
                resolved_host.instances.append(ResolverInstance(instance))
        return resolved_host

    def resolve_host(self, host: str) -> ResolverHost:
        try:
            data = self.get_json('/v1/host_instances/{host}'.format(host=host))
            return self.parse_resolver_host(data)
        except Exception as ex:
            raise Exception("Error in rtc-resolver request {0}".format(str(ex)))
