from infra.rtc.jyggalag.clients.walle_host import WalleHost
from infra.rtc.jyggalag.jyggalag_config import JyggalagConfig
from urllib.parse import quote_plus
from requests import request
from time import time
import json
import re

default_host_fields = 'inv,name,project,state,status,location.short_datacenter_name,task.status,task.status_message,' \
                      'task.owner,ticket,task.error'
default_log_fields = 'id,host_inv,host_name,time,status_time,project,status,issuer,type,reason,error,payload'


class WalleApi:

    def __init__(self, config: JyggalagConfig = None):
        self.api_host = config.walle.get('api_host', 'https://api.wall-e.yandex-team.ru')
        self.token = config.walle.get('oauth_token', None)

    def prep_request(self, method: str, url: str, data=None, headers: dict = None, args: dict = None):
        if headers is None:
            headers = dict()
        if args is not None and len(args) > 0:
            argvalues = list()
            for key in args.keys():
                argvalues.append('{key}={value}'.format(key=quote_plus(key), value=quote_plus(args[key])))
            url = "{url}?{args}".format(url=url, args='&'.join(argvalues))
        if self.token is not None:
            headers['Authorization'] = 'OAuth {auth}'.format(auth=self.token)
        return request(
            method=method,
            url='{host}{url}'.format(host=self.api_host, url=url),
            verify=False,
            data=data,
            headers=headers
        )

    def add_host(self, host_id: str, project: str, comment: str, instant: bool):
        inv = None
        name = None
        if re.match(r'[0-9]+', host_id):
            inv = int(host_id)
        else:
            name = host_id
        data = {
            'project': project,
            'reason': comment,
            'state': 'assigned',
            'status': 'ready',
            'instant': instant
        }
        if inv is None:
            data['name'] = name
        else:
            data['inv'] = inv
        req = self.prep_request(
            method='POST',
            url='/v1/hosts',
            headers={'Content-Type': 'application/json'},
            data=json.dumps(data)
        )
        req.raise_for_status()
        return req.text

    def prepare_host(self, host: WalleHost, comment: str):
        req = self.prep_request(
            'POST',
            '/v1/hosts/{inv}/prepare'.format(inv=host.inv),
            headers={'Content-Type': 'application/json'},
            data=json.dumps({'reason': comment, 'ignore_cms': True, 'with_auto_healing': True})
        )
        req.raise_for_status()
        return req.text

    def release_host(self, host: WalleHost, project: str, comment: str):
        req = self.prep_request(
            'POST',
            '/v1/hosts/{inv}/switch-project'.format(inv=str(host.inv)),
            headers={'Content-Type': 'application/json'},
            data=json.dumps({
                'reason': comment,
                'project': project,
                'ignore_cms': True,
                'release': True,
                'erase_disks': True
            }),
            args={'ignore_maintenance': 'true'}
        )
        req.raise_for_status()
        return req.text

    def force_switch_project(self, host: WalleHost, project: str, comment: str):
        req = self.prep_request(
            'POST',
            '/v1/hosts/{inv}/switch-project'.format(inv=str(host.inv)),
            headers={'Content-Type': 'application/json'},
            data=json.dumps({
                'reason': comment,
                'project': project,
                'ignore_cms': True,
                'release': False,
                'force': True,
                'erase_disks': False,
                'disable_admin_requests': True
            }),
            args={'ignore_maintenance': 'true'}
        )
        req.raise_for_status()
        return req.text

    def force_status(self, host: WalleHost, status: str, comment: str):
        req = self.prep_request(
            method='POST',
            url='/v1/hosts/{inv}/force-status'.format(inv=str(host.inv)),
            headers={'Content-Type': 'application/json'},
            data=json.dumps({'reason': comment, 'status': status}),
            args={'ignore_maintenance': 'true'}
        )
        req.raise_for_status()
        return req.text

    def power_off(self, host: WalleHost, comment: str):
        req = self.prep_request(
            'POST',
            '/v1/hosts/{inv}/set-maintenance'.format(inv=str(host.inv)),
            headers={'Content-Type': 'application/json'},
            data=json.dumps({
                'ignore_cms': True,
                'power_off': True,
                'reason': comment,
                'timeout_time': int(time()) + 20 * 24 * 3600
            }),
        )
        req.raise_for_status()
        return req.text

    def set_maintenance(self, host: WalleHost, comment: str):
        req = self.prep_request(
            'POST',
            '/v1/hosts/{inv}/set-maintenance'.format(inv=str(host.inv)),
            headers={'Content-Type': 'application/json'},
            data=json.dumps({'reason': comment, 'timeout_time': int(time()) + 20 * 24 * 3600}),
        )
        req.raise_for_status()
        return req.text

    def get_hosts(self, host_ids: list):
        invs = list()
        names = list()
        for host_id in host_ids:
            if re.match(r'[0-9]+', host_id):
                invs.append(int(host_id))
            else:
                names.append(host_id)

        data = dict()
        data['invs'] = invs
        data['names'] = names

        data_json = json.dumps(data)
        req = self.prep_request(
            method='POST',
            url='/v1/get-hosts',
            args={'fields': default_host_fields, 'limit': '10000'},
            headers={'Content-Type': 'application/json'},
            data=data_json,
        )
        req.raise_for_status()
        data_res = req.json()
        result = list()

        for walle_data in data_res['result']:
            result.append(WalleHost(walle_data))
        return result

    def get_host(self, host_id: str):
        req = self.prep_request(
            method='GET',
            url='/v1/hosts/{host}'.format(host=host_id),
            args={'fields': default_host_fields}
        )
        req.raise_for_status()
        return WalleHost(req.json())

    def get_audit_log(self, host_id: str):
        find_field = 'host_name'
        if re.match(r'[0-9]+', host_id):
            find_field = 'host_inv'

        req = self.prep_request(
            method='GET',
            url='/v1/audit-log',
            args={'fields': default_log_fields, find_field: host_id, 'limit': '100'}
        )
        req.raise_for_status()
        return req.json()
