from infra.rtc.jyggalag.jyggalag_host import JyggalagHost
from infra.rtc.jyggalag.jyggalag_log import JyggalagLog
from infra.rtc.jyggalag.jyggalag_config import JyggalagConfig
from time import time
import psycopg2.pool
import socket


allowed_states = {
    'QLOUD_REDEPLOY_QUEUED',
    'QLOUD_REDEPLOY_RELEASE_PENDING',
    'QLOUD_REDEPLOY_RELEASING',
    'QLOUD_REDEPLOY_ERASING',
    'QLOUD_REDEPLOY_PREPARING',
    'QLOUD_REDEPLOY_ADDING',
    'QLOUD_REDEPLOY_BUFFER_PENDING_BUSY',
    'QLOUD_REDEPLOY_BUFFER_BUSY',
    'QLOUD_REDEPLOY_BUFFER_FREE',
    'QLOUD_REDEPLOY_BUFFER_RELEASE_PENDING',
    'QLOUD_REDEPLOY_BUFFER_RELEASING',
    'QLOUD_MOVE_QUEUED',
    'QLOUD_MOVE_RELEASE_PENDING',
    'QLOUD_MOVE_RELEASING',
    'QLOUD_ADD_QUEUED',
    'QLOUD_ADD_POWER_OFF',
    'QLOUD_ADD_ERASING',
    'QLOUD_ADD_PREPARING',
    'QLOUD_ADD_ADDING',
    'ADD_QUEUED',
    'ADD_POWER_OFF',
    'ADD_ERASING',
    'ADD_PREPARING',
    'ADD_ADDING',
    'FINISHED'
}

pending_states = {
    'QLOUD_REDEPLOY_RELEASE_PENDING',
    'QLOUD_REDEPLOY_BUFFER_RELEASE_PENDING',
    'QLOUD_MOVE_RELEASE_PENDING'
}

releasing_states = {
    'QLOUD_REDEPLOY_RELEASING',
    'QLOUD_REDEPLOY_BUFFER_RELEASING',
    'QLOUD_MOVE_RELEASING'
}

states_need_ok = {
    'QLOUD_REDEPLOY_RELEASE_PENDING',
    'QLOUD_REDEPLOY_BUFFER_RELEASE_PENDING',
    'QLOUD_MOVE_RELEASE_PENDING'
}


class JyggalagData:

    def __init__(self, config: JyggalagConfig):
        self.pg_pool = psycopg2.pool.ThreadedConnectionPool(
            1,
            int(config.db.get('connections', 32)),
            database=str(config.db.get('database', None)),
            user=str(config.db.get('username', None)),
            password=str(config.db.get('password', None)),
            host=str(config.db.get('host', None)),
            port=int(config.db.get('port', 6432)),
            target_session_attrs='read-write'
        )
        self.lock_timeout = int(config.db.get('lock_timeout', 32))
        self.init_db()

    def get_lock(self, resource: str = '*'):
        if resource is None:
            return True
        current_time = int(time())
        hostname = socket.gethostname()
        conn = self.pg_pool.getconn()
        try:
            # Deleting old
            with conn.cursor() as cur:
                cur.execute(
                    "DELETE FROM locks WHERE time <= %s and resource = %s;",
                    (current_time - self.lock_timeout, resource)
                )
            conn.commit()

            # Prechecking
            with conn.cursor() as cur:
                cur.execute("SELECT hostname FROM locks WHERE resource = %s;", (resource, ))
                lock = cur.fetchall()
                if len(lock) != 0:
                    self.pg_pool.putconn(conn)
                    return False

            # Getting lock
            with conn.cursor() as cur:
                cur.execute(
                    "INSERT INTO locks (hostname, resource, time) VALUES (%s, %s, %s);",
                    (hostname, resource, current_time)
                )
            conn.commit()

            # Checking lock
            with conn.cursor() as cur:
                cur.execute("SELECT hostname FROM locks WHERE resource = %s;", (resource, ))
                lock = cur.fetchall()
                if len(lock) == 1 and lock[0][0] == hostname:
                    return True
                else:
                    return False
        finally:
            if conn is not None:
                self.pg_pool.putconn(conn)

    def release_lock(self, resource: str = '*'):
        if resource is None:
            return
        conn = self.pg_pool.getconn()
        try:
            with conn.cursor() as cur:
                cur.execute(
                    "DELETE FROM locks WHERE hostname = %s and resource = %s;",
                    (socket.gethostname(), resource)
                )
            conn.commit()
        finally:
            if conn is not None:
                self.pg_pool.putconn(conn)

    def init_db(self):
        conn = self.pg_pool.getconn()
        try:
            with conn.cursor() as cur:
                cur.execute(
                    "CREATE TABLE IF NOT EXISTS locks(\n"
                    "hostname VARCHAR(256),\n"
                    "resource VARCHAR(256),\n"
                    "time bigint\n"
                    ");"
                )
                cur.execute(
                    "CREATE TABLE IF NOT EXISTS hosts(\n"
                    "inv bigint UNIQUE, \n"
                    "name VARCHAR(128) UNIQUE,\n"
                    "state VARCHAR(64) NOT NULL,\n"
                    "update_time bigint,\n"
                    "ok_time bigint,\n"
                    "ticket VARCHAR(128),\n"
                    "qloud_segment VARCHAR(128),\n"
                    "walle_project VARCHAR(128)\n"
                    ");")

                cur.execute(
                    "CREATE TABLE IF NOT EXISTS log(\n"
                    "inv bigint,\n"
                    "name VARCHAR(128),\n"
                    "node_name VARCHAR(128),\n"
                    "state_from VARCHAR(128),\n"
                    "state_to VARCHAR(128),\n"
                    "time bigint\n"
                    ");")
            conn.commit()
        finally:
            if conn is not None:
                self.pg_pool.putconn(conn)

    def update_project(self, host: JyggalagHost, project: str):
        conn = self.pg_pool.getconn()
        try:
            with conn.cursor() as cur:
                if host.inv is None:
                    cur.execute(
                        "UPDATE hosts SET walle_project = %s WHERE name = %s;",
                        (project, host.name)
                    )
                else:
                    cur.execute(
                        "UPDATE hosts SET walle_project = %s WHERE inv = %s;",
                        (project, host.inv)
                    )
            conn.commit()
        finally:
            if conn is not None:
                self.pg_pool.putconn(conn)

    def update_segment(self, host: JyggalagHost, segment: str):
        conn = self.pg_pool.getconn()
        try:
            with conn.cursor() as cur:
                if host.inv is None:
                    cur.execute(
                        "UPDATE hosts SET qloud_segment = %s WHERE name = %s;",
                        (segment, host.name)
                    )
                else:
                    cur.execute(
                        "UPDATE hosts SET qloud_segment = %s WHERE inv = %s;",
                        (segment, host.inv)
                    )
            conn.commit()
        finally:
            if conn is not None:
                self.pg_pool.putconn(conn)

    def get_hosts(self):
        conn = self.pg_pool.getconn()
        try:
            result = list()
            with conn.cursor() as cur:
                cur.execute(
                    "SELECT inv, name, state, update_time, qloud_segment, walle_project, ticket, ok_time FROM hosts;"
                )
                for result_row in cur.fetchall():
                    if result_row[2] is not None and str(result_row[2]) not in allowed_states:
                        raise Exception('Invalid state in database for {name}: {state}'.format(
                            state=str(result_row[2]),
                            name=str(result_row[1])
                        ))
                    result.append(JyggalagHost(result_row))
            return result
        finally:
            if conn is not None:
                self.pg_pool.putconn(conn)

    def update_inv(self, inv: int, name: str):
        conn = self.pg_pool.getconn()
        try:
            with conn.cursor() as cur:
                cur.execute(
                    "UPDATE hosts SET inv = %s WHERE name = %s;",
                    (inv, name)
                )
            conn.commit()
        finally:
            if conn is not None:
                self.pg_pool.putconn(conn)

    def set_error(self, error: str, inv: int, name: str):
        conn = self.pg_pool.getconn()
        try:
            with conn.cursor() as cur:
                if inv is None and name is not None:
                    cur.execute(
                        "UPDATE hosts SET last_error = %s WHERE name = %s;",
                        (error, name)
                    )
                elif name is None and inv is not None:
                    cur.execute(
                        "UPDATE hosts SET last_error = %s WHERE inv = %s;",
                        (error, inv)
                    )
            conn.commit()
        finally:
            if conn is not None:
                self.pg_pool.putconn(conn)

    def update_hostname(self, inv: int, name: str):
        conn = self.pg_pool.getconn()
        try:
            with conn.cursor() as cur:
                cur.execute(
                    "UPDATE hosts SET name = %s WHERE inv = %s;",
                    (name, inv)
                )
            conn.commit()
        finally:
            if conn is not None:
                self.pg_pool.putconn(conn)

    def ok_all(self):
        conn = self.pg_pool.getconn()
        try:
            with conn.cursor() as cur:
                cur.execute(
                    "UPDATE hosts SET ok_time = %s;",
                    (int(time()), )
                )
            conn.commit()
        finally:
            if conn is not None:
                self.pg_pool.putconn(conn)

    def ok_host(self, inv: int = None, name: str = None):
        conn = self.pg_pool.getconn()
        try:
            with conn.cursor()as cur:
                if inv is None and name is not None:
                    cur.execute(
                        "UPDATE hosts SET ok_time = %s WHERE name = %s;",
                        (int(time()), name)
                    )
                elif name is None and inv is not None:
                    cur.execute(
                        "UPDATE hosts SET ok_time = %s WHERE inv = %s;",
                        (int(time()), inv)
                    )
            conn.commit()
        finally:
            if conn is not None:
                self.pg_pool.putconn(conn)

    def remove_host(self, host: JyggalagHost):
        conn = self.pg_pool.getconn()
        try:
            with conn.cursor() as cur:
                if host.inv is not None:
                    cur.execute(
                        "DELETE FROM hosts WHERE inv = %s;",
                        (host.inv, )
                    )
                elif host.name is not None:
                    cur.execute(
                        "DELETE FROM hosts WHERE name = %s;",
                        (host.name, )
                    )
                else:
                    raise Exception('NULL host')
            conn.commit()
        finally:
            if conn is not None:
                self.pg_pool.putconn(conn)

    def set_ticket(self, ticket: str, inv: int = None, name: str = None):
        conn = self.pg_pool.getconn()
        try:
            with conn.cursor() as cur:
                if inv is None and name is not None:
                    cur.execute(
                        "UPDATE hosts SET ticket = %s, update_time = %s, ok_time = %s WHERE name = %s;",
                        (ticket, int(time()), 0, name)
                    )
                elif name is None and inv is not None:
                    cur.execute(
                        "UPDATE hosts SET ticket = %s, update_time = %s, ok_time = %s WHERE inv = %s;",
                        (ticket, int(time()), 0, inv)
                    )
            conn.commit()
        finally:
            if conn is not None:
                self.pg_pool.putconn(conn)

    def add_host(self, name: str, segment: str, state: str):
        if state not in allowed_states:
            raise Exception('Invalid state for {name}: {state}'.format(
                state=str(state),
                name=str(name)
            ))
        conn = self.pg_pool.getconn()
        try:
            with conn.cursor() as cur:
                cur.execute(
                    "INSERT INTO hosts (name, qloud_segment, state, inv, update_time) VALUES(%s,%s,%s,%s,%s);",
                    (name, segment, state, None, int(time()))
                )
            conn.commit()
        finally:
            if conn is not None:
                self.pg_pool.putconn(conn)

    def get_log(self, delta: int = 3600 * 24 * 7):
        current_time = int(time())
        conn = self.pg_pool.getconn()
        try:
            res = list()
            with conn.cursor() as cur:
                cur.execute(
                    "SELECT time, inv, name, node_name, state_from, state_to FROM log\n"
                    "WHERE time NOTNULL AND time >= %s ORDER BY time;",
                    (current_time - delta, )
                )
                for row in cur.fetchall():
                    res.append(JyggalagLog(row))
            return res
        finally:
            if conn is not None:
                self.pg_pool.putconn(conn)

    def last_operation(self) -> int:
        conn = self.pg_pool.getconn()
        try:
            with conn.cursor() as cur:
                cur.execute(
                    "SELECT MAX(time) FROM log\n"
                    "WHERE time NOTNULL;"
                )
                res = cur.fetchone()
                if res is not None:
                    res = int(res[0])
                else:
                    res = 0
                return res
        finally:
            if conn is not None:
                self.pg_pool.putconn(conn)

    def last_release(self):
        conn = self.pg_pool.getconn()
        try:
            with conn.cursor() as cur:
                cur.execute(
                    "SELECT MAX(time) FROM log\n"
                    "WHERE time NOTNULL AND\n"
                    "(state_to = 'QLOUD_REDEPLOY_BUFFER_RELEASING' OR\n"
                    "state_to = 'QLOUD_MOVE_RELEASING' OR\n"
                    "state_to = 'QLOUD_REDEPLOY_RELEASING');"
                )
                res = cur.fetchone()
                if res is not None:
                    res = int(res[0])
                else:
                    res = 0
                return res
        finally:
            if conn is not None:
                self.pg_pool.putconn(conn)

    def get_envs_metrics(self):
        conn = self.pg_pool.getconn()
        result = dict()
        for state in allowed_states:
            result[state] = 0
            if state in states_need_ok:
                result['{state}_OK'.format(state=state)] = 0
                result['{state}_WAIT'.format(state=state)] = 0
        try:
            with conn.cursor() as cur:
                cur.execute(
                    "SELECT state, ok_time, update_time FROM hosts;",
                )
                for row in cur.fetchall():
                    state = str(row[0])
                    result[state] += 1
                    if state in states_need_ok:
                        if row[1] is not None and row[2] is not None and int(row[1]) > int(row[2]):
                            result['{state}_OK'.format(state=state)] += 1
                        else:
                            result['{state}_WAIT'.format(state=state)] += 1
            return result
        finally:
            if conn is not None:
                self.pg_pool.putconn(conn)

    def get_state_metrics(self):
        conn = self.pg_pool.getconn()
        try:
            result = dict()
            for state in allowed_states:
                result[state] = 0
                if state in states_need_ok:
                    result['{state}_OK'.format(state=state)] = 0
                    result['{state}_WAIT'.format(state=state)] = 0
            with conn.cursor() as cur:
                cur.execute(
                    "SELECT state, ok_time, update_time FROM hosts;",
                )
                for row in cur.fetchall():
                    state = str(row[0])
                    result[state] += 1
                    if state in states_need_ok:
                        if row[1] is not None and row[2] is not None and int(row[1]) > int(row[2]):
                            result['{state}_OK'.format(state=state)] += 1
                        else:
                            result['{state}_WAIT'.format(state=state)] += 1
            return result
        finally:
            if conn is not None:
                self.pg_pool.putconn(conn)

    def write_log(self, inv: int, name: str, from_state: str, to_state: str):
        hostname = socket.gethostname()
        conn = self.pg_pool.getconn()
        try:
            with conn.cursor() as cur:
                cur.execute(
                    "INSERT INTO log (inv, name, node_name, state_from, state_to, time) VALUES(%s,%s,%s,%s,%s,%s);",
                    (int(inv), str(name), str(hostname), str(from_state), str(to_state), int(time()))
                )
            conn.commit()
        finally:
            if conn is not None:
                self.pg_pool.putconn(conn)

    def update_host_state(self, state: str, inv: int = None, name: str = None):
        if state not in allowed_states:
            raise Exception('Invalid state for {name}: {state}'.format(
                state=str(state),
                name=str(name)
            ))
        conn = self.pg_pool.getconn()
        try:
            with conn.cursor() as cur:
                if name is not None:
                    cur.execute(
                        "UPDATE hosts SET state = %s, update_time = %s, ok_time = %s WHERE name = %s;",
                        (state, int(time()), 0, name)
                    )

                elif inv is not None:
                    cur.execute(
                        "UPDATE hosts SET state = %s, update_time = %s, ok_time = %s WHERE inv = %s;",
                        (state, int(time()), 0, inv)
                    )
                else:
                    raise Exception('no name or inv to change state')
            conn.commit()
        finally:
            if conn is not None:
                self.pg_pool.putconn(conn)
