from infra.rtc.jyggalag.clients.service_resolver import ServiceResolver
from infra.rtc.jyggalag.clients.qloud_api import QloudApi
from infra.rtc.jyggalag.clients.qloud_api import down_states as qloud_down_states, up_states as qloud_up_states
from infra.rtc.jyggalag.clients.walle_api import WalleApi
from infra.rtc.jyggalag.jyggalag_host import JyggalagHost
from infra.rtc.jyggalag.jyggalag_config import JyggalagConfig
from infra.rtc.jyggalag.jyggalag_data import JyggalagData
from datetime import datetime
from time import time


pending_states = {
    'QLOUD_REDEPLOY_RELEASE_PENDING',
    'QLOUD_REDEPLOY_BUFFER_RELEASE_PENDING',
    'QLOUD_MOVE_RELEASE_PENDING'
}

releasing_states = {
    'QLOUD_REDEPLOY_RELEASING',
    'QLOUD_REDEPLOY_BUFFER_RELEASING',
    'QLOUD_MOVE_RELEASING'
}

states_need_ok = {
    'QLOUD_REDEPLOY_RELEASE_PENDING',
    'QLOUD_REDEPLOY_BUFFER_RELEASE_PENDING',
    'QLOUD_MOVE_RELEASE_PENDING'
}


class JyggalagMachine:

    def __init__(
            self,
            config: JyggalagConfig,
            qloud: QloudApi,
            walle: WalleApi,
            resolver: ServiceResolver,
            database: JyggalagData):
        self.database = database
        self.qloud = qloud
        self.walle = walle
        self.resolver = resolver
        self.logger = config.logger
        self.use_resolver = config.redeploy.get('resolver_check', True)
        self.junk_project = config.redeploy.get('junk_project', 'search-delete')
        self.operation_period = int(config.redeploy.get('operation_period', 300))
        self.project_map = config.redeploy.get('project_map', None)
        self.buffer_segment_name = str(config.redeploy.get('buffer_segment_name', 'gateway'))
        self.readonly = config.readonly
        self.environments_metrics = dict()
        self.last_host_errors = 0
        self.last_global_errors = 0
        self.can_operate = False
        self.poweroff_timeout = int(config.redeploy.get('poweroff_timeout', 86400))

    def get_hosts(self):
        hosts = self.database.get_hosts()

        hostnames = list()
        hostinvs = list()
        for host in hosts:
            if host.name is not None:
                hostnames.append(host.name)
            if host.inv is not None:
                hostinvs.append(str(host.inv))

        hosts_qloud_names = self.qloud.find_hosts(hostnames)
        if (
                len(hosts_qloud_names) == 0
        ):
            raise Exception('Fatal Qloud Hosts Count')

        hosts_walle_names = self.walle.get_hosts(hostnames)
        hosts_walle_invs = self.walle.get_hosts(hostinvs)

        qloud_hosts_name_map = dict()
        walle_hosts_name_map = dict()
        walle_hosts_inv_map = dict()

        for qloud_host in hosts_qloud_names:
            qloud_hosts_name_map[qloud_host.name] = qloud_host

        for walle_host in hosts_walle_names:
            walle_hosts_name_map[walle_host.name] = walle_host

        for walle_host in hosts_walle_invs:
            walle_hosts_inv_map[walle_host.inv] = walle_host

        uniq_envs_pending = set()
        uniq_envs_pending_ok = set()
        uniq_envs_releasing = set()
        for host in hosts:
            if host.name in qloud_hosts_name_map:
                host.qloud = qloud_hosts_name_map[host.name]
                if host.state in pending_states:
                    for slot in host.qloud.services.keys():
                        uniq_envs_pending.add(host.qloud.services[slot].get_full_env_name())
                        if host.is_ok():
                            uniq_envs_pending_ok.add(host.qloud.services[slot].get_full_env_name())
                if host.state in releasing_states:
                    for slot in host.qloud.services.keys():
                        uniq_envs_releasing.add(host.qloud.services[slot].get_full_env_name())

            if host.inv is not None and host.inv in walle_hosts_inv_map:
                host.walle = walle_hosts_inv_map[host.inv]
                if host.name != host.walle.name:

                    self.logger.info('Renaming [{inv_old}]{name_old} to [{inv_new}]{name_new}'.format(
                        inv_old=host.inv,
                        name_old=host.name,
                        inv_new=host.walle.inv,
                        name_new=host.walle.name
                    ))
                    self.database.update_hostname(int(host.walle.inv), host.walle.name)
                    host.name = host.walle.name
            elif host.name in walle_hosts_name_map:
                host.walle = walle_hosts_name_map[host.name]
                if host.inv != host.walle.inv:
                    self.logger.info('Renaming [{inv_old}]{name_old} to [{inv_new}]{name_new}'.format(
                        inv_old=host.inv,
                        name_old=host.name,
                        inv_new=host.walle.inv,
                        name_new=host.walle.name
                    ))
                    self.database.update_inv(int(host.walle.inv), host.walle.name)
                    host.inv = int(host.walle.inv)
        self.environments_metrics = {
            'PENDING_RELEASE': len(uniq_envs_pending),
            'PENDING_RELEASE_OK': len(uniq_envs_pending_ok),
            'RELEASING': len(uniq_envs_releasing)
        }
        return hosts

    def _spend_operation(self):
        if self.can_operate:
            self.can_operate = False
            return True
        return False

    def get_pre_project(self, target_project: str):
        if self.project_map is None:
            return target_project
        if target_project in self.project_map:
            return self.project_map[target_project]
        return target_project

    def tick(self):
        self.last_host_errors = 0
        self.last_global_errors = 0
        current_time = time()
        last_op = self.database.last_operation()
        self.can_operate = (current_time - last_op) >= self.operation_period
        if self.readonly:
            return
        try:
            for host in self.get_hosts():
                try:
                    self.process_host(host)
                except Exception:
                    self.logger.exception('Error while processing host {}'.format(host))
                    self.last_host_errors += 1
        except Exception:
            self.logger.exception('Error while getting hosts')
            self.last_global_errors += 1

    def process_host(self, host: JyggalagHost):
        if host.state == 'QLOUD_REDEPLOY_QUEUED':
            if self.qloud_host_released(host, None):
                if self._spend_operation():
                    self.switch_state(host, 'QLOUD_REDEPLOY_RELEASING')
        elif host.state == 'QLOUD_REDEPLOY_RELEASE_PENDING':
            if self.qloud_host_released(host, 'SUSPECTED'):
                if self._spend_operation():
                    self.switch_state(host, 'QLOUD_REDEPLOY_RELEASING')
            elif host.qloud is not None and host.qloud.state in qloud_down_states:
                if self._spend_operation():
                    self.switch_state(host, 'QLOUD_REDEPLOY_RELEASING')
            elif host.is_ok():
                if self._spend_operation():
                    self.switch_state(host, 'QLOUD_REDEPLOY_RELEASING')
        elif host.state == 'QLOUD_REDEPLOY_RELEASING':
            if self.release_and_remove_qloud_host(host):
                if self._spend_operation():
                    self.switch_state(host, 'QLOUD_REDEPLOY_ERASING')
        elif host.state == 'QLOUD_REDEPLOY_ERASING':
            target_project = self.qloud.get_segment_walle_project(host.segment)
            if self.qloud_host_erased(host, target_project):
                if self._spend_operation():
                    self.switch_state(host, 'QLOUD_REDEPLOY_PREPARING')
        elif host.state == 'QLOUD_REDEPLOY_PREPARING':
            target_project = self.qloud.get_segment_walle_project(host.segment)
            if self.qloud_host_prepared(host, target_project):
                if self._spend_operation():
                    self.switch_state(host, 'QLOUD_REDEPLOY_ADDING')
        elif host.state == 'QLOUD_REDEPLOY_ADDING':
            if self.host_added_to_qloud(host, host.segment):
                if self._spend_operation():
                    self.switch_state(host, 'FINISHED')
        elif host.state == 'QLOUD_REDEPLOY_BUFFER_FREE':
            self.host_added_to_qloud(host, host.segment, 'DOWN')
        elif host.state == 'QLOUD_REDEPLOY_BUFFER_PENDING_BUSY':
            if self.host_added_to_qloud(host, host.segment, 'UP'):
                self.switch_state(host, 'QLOUD_REDEPLOY_BUFFER_BUSY')
        elif host.state == 'QLOUD_REDEPLOY_BUFFER_BUSY':
            if host.qloud.segment.split('.')[1] == self.buffer_segment_name:
                self.database.update_segment(host, host.qloud.segment)
                self.switch_state(host, 'QLOUD_REDEPLOY_BUFFER_FREE')
            else:
                self.host_in_target_segment(host, host.segment, 'UP')
        elif host.state == 'QLOUD_REDEPLOY_BUFFER_RELEASE_PENDING':
            if self.qloud_host_released(host, 'SUSPECTED'):
                if self._spend_operation():
                    self.switch_state(host, 'QLOUD_REDEPLOY_BUFFER_RELEASING')
            elif host.qloud is not None and host.qloud.state in qloud_down_states:
                if self._spend_operation():
                    self.switch_state(host, 'QLOUD_REDEPLOY_BUFFER_RELEASING')
            elif host.is_ok():
                if self._spend_operation():
                    self.switch_state(host, 'QLOUD_REDEPLOY_BUFFER_RELEASING')
        elif host.state == 'QLOUD_REDEPLOY_BUFFER_RELEASING':
            if self.qloud_host_released(host, 'DOWN'):
                buffer_segment = '{}.{}'.format(host.qloud.installation, self.buffer_segment_name)
                if self.host_in_target_segment(host, buffer_segment, 'DOWN'):
                    if self._spend_operation():
                        self.database.update_segment(host, buffer_segment)
                        self.switch_state(host, 'QLOUD_REDEPLOY_BUFFER_FREE')
        elif host.state == 'QLOUD_MOVE_QUEUED':
            if self._spend_operation():
                self.switch_state(host, 'QLOUD_MOVE_RELEASE_PENDING')
        elif host.state == 'QLOUD_MOVE_RELEASE_PENDING':
            if host.is_ok():
                if self._spend_operation():
                    self.switch_state(host, 'QLOUD_MOVE_RELEASING')
        elif host.state == 'QLOUD_MOVE_RELEASING':
            if self.host_added_to_qloud(host, host.segment):
                self.switch_state(host, 'FINISHED')
        elif host.state == 'QLOUD_ADD_QUEUED':
            self.switch_state(host, 'QLOUD_ADD_POWER_OFF')
        elif host.state == 'QLOUD_ADD_POWER_OFF':
            current_time = int(time())
            if self.host_powered_off(host) and current_time - host.update_time >= self.poweroff_timeout:
                self.switch_state(host, 'QLOUD_ADD_ERASING')
        elif host.state == 'QLOUD_ADD_ERASING':
            target_project = self.qloud.get_segment_walle_project(host.segment)
            if self.qloud_host_erased(host, target_project):
                self.switch_state(host, 'QLOUD_ADD_PREPARING')
        elif host.state == 'QLOUD_ADD_PREPARING':
            target_project = self.qloud.get_segment_walle_project(host.segment)
            if self.qloud_host_prepared(host, target_project):
                self.switch_state(host, 'QLOUD_ADD_ADDING')
        elif host.state == 'QLOUD_ADD_ADDING':
            if self.host_added_to_qloud(host, host.segment):
                self.switch_state(host, 'FINISHED')
        elif host.state == 'ADD_QUEUED':
            self.switch_state(host, 'ADD_POWER_OFF')
        elif host.state == 'ADD_POWER_OFF':
            current_time = int(time())
            if self.host_powered_off(host) and current_time - host.update_time >= self.poweroff_timeout:
                self.switch_state(host, 'ADD_ERASING')
        elif host.state == 'ADD_ERASING':
            pre_project = self.get_pre_project(host.walle_project)
            if self.qloud_host_erased(host, pre_project):
                self.switch_state(host, 'ADD_PREPARING')
        elif host.state == 'ADD_PREPARING':
            pre_project = self.get_pre_project(host.walle_project)
            if self.qloud_host_prepared(host, pre_project):
                self.switch_state(host, 'ADD_ADDING')
        elif host.state == 'ADD_ADDING':
            if self.host_switched_to_project(host, host.walle_project):
                self.switch_state(host, 'FINISHED')
        elif host.state == 'FINISHED':
            self.database.remove_host(host)

    def qloud_host_moved(self, host: JyggalagHost, target_segment: str) -> bool:
        target_project = self.qloud.get_segment_walle_project(target_segment)
        if not self.qloud_host_prepared(host, target_project):
            return False
        if host.qloud is None:
            self.qloud.add_host(host.name, target_segment)
            return False
        if host.qloud.segment == target_segment:
            return True
        return False

    def host_in_target_segment(self, host: JyggalagHost, target_segment: str, target_state='UP') -> bool:
        if target_segment is None:
            raise Exception('Target segment is none')
        if host.qloud is None:
            self.logger.info('{} Adding host to qloud, segment {}'.format(host, target_segment))
            self.qloud.add_host(host.name, target_segment)
            return False
        if host.qloud.segment != target_segment:
            if self.qloud_host_released(host):
                self.logger.info('{} Switching segment from {} to {}'.format(host, host.qloud.segment, target_segment))
                self.qloud.change_host_segment(
                    host.qloud,
                    target_segment,
                    'SWITCH SEGMENT FROM {} TO {} [{}]'.format(host.qloud.segment, target_segment, datetime.now())
                )
        if host.qloud.state != target_state:
            if host.qloud.state not in self.qloud.ignore_up_states or target_state not in qloud_up_states:
                self.logger.info('{} changing host state from {} to {}'.format(host, host.qloud.state, target_state))
                self.qloud.set_host_state(host.qloud, target_state, 'WRONG TARGET STATE {}'.format(datetime.now()))
                host.qloud.state = target_state
            return False
        return True

    def host_added_to_qloud(self, host: JyggalagHost, target_segment: str, target_state: str = 'UP') -> bool:
        if target_segment is None:
            raise Exception('Target segment is none')
        target_project = self.qloud.get_segment_walle_project(target_segment)
        if not self.qloud_host_prepared(host, target_project):
            return False
        if host.qloud is None:
            self.logger.info('{} Adding host to qloud, segment {}'.format(host, target_segment))
            self.qloud.add_host(host.name, target_segment)
            return False
        if host.qloud.is_empty_data:
            self.qloud.update_metas([host.qloud])
            return False
        if host.qloud.segment != target_segment:
            if self.qloud_host_released(host):
                self.logger.info('{} Switching segment from {} to {}'.format(host, host.qloud.segment, target_segment))
                self.qloud.change_host_segment(
                    host.qloud,
                    target_segment,
                    'SWITCH SEGMENT FROM {} TO {} [{}]'.format(host.qloud.segment, target_segment, datetime.now())
                )
        if target_state is None:
            target_state = 'UP'
        if host.qloud.state != target_state:
            if host.qloud.state not in self.qloud.ignore_up_states:
                self.logger.info('{} Setting state UP in segment {}'.format(host, host.qloud.segment))
                self.qloud.set_host_state(host.qloud, 'UP', 'COMMISSION NEW HOST [{}]'.format(datetime.now()))
            return False
        return True

    def host_switched_to_project(self, host: JyggalagHost, target_project: str) -> bool:
        if target_project is None:
            raise Exception('Target project is none')
        if host.walle is None:
            return False
        elif host.walle.project != target_project:
            self.logger.info('{} Switching from project {} to {}'.format(host, host.walle.project, target_project))
            self.walle.force_switch_project(host.walle, target_project, 'FORCE MOVE TO TARGET PROJECT {}'.format(
                target_project
            ))
            return False
        return True

    def qloud_host_prepared(self, host: JyggalagHost, target_project: str, soft: bool = False) -> bool:
        if target_project is None:
            raise Exception('Target project is none')
        if host.walle is None:
            host_id = str(host.inv)
            if host_id is None:
                host_id = host.name
            self.logger.info('{} Adding host to wall-e'.format(host))
            self.walle.add_host(host_id, self.junk_project, 'ADDING HOST FOR PREPARING', True)
            return False
        elif host.walle.project == target_project and host.walle.state == 'assigned' and host.walle.status == 'ready':
            return True
        elif host.walle.project == target_project and host.walle.state == 'free' and host.walle.status == 'ready':
            if not soft:
                self.logger.info('{} Preparing host in wall-e'.format(host))
                self.walle.prepare_host(host.walle, 'PREPARING HOST')
            return False
        elif host.walle.project != target_project and host.walle.status in {'ready', 'dead', 'manual'}:
            if not soft:
                self.release_remove_and_erase_qloud_host(host, target_project)
        return False

    def host_powered_off(self, host: JyggalagHost, soft: bool = False):
        if host.walle is None:
            host_id = str(host.inv)
            if host_id is None:
                host_id = host.name
            if not soft:
                self.walle.add_host(host_id, self.junk_project, 'ADDING HOST FOR ERASING', True)
            return False
        if host.walle.state != 'maintenance':
            if host.walle.status in {'ready', 'dead', 'manual'}:
                self.walle.power_off(host.walle, 'Commission power off')
            return False
        return True

    def qloud_host_erased(self, host: JyggalagHost, target_project: str, soft: bool = False):
        if target_project is None:
            raise Exception('Target project is none')
        if host.walle is None:
            host_id = str(host.inv)
            if host_id is None:
                host_id = host.name
            if not soft:
                self.walle.add_host(host_id, self.junk_project, 'ADDING HOST FOR ERASING', True)
            return False
        if host.walle.project == target_project and host.walle.state == 'free' and host.walle.status == 'ready':
            return True
        if host.walle.project != target_project and host.walle.status in {'ready', 'dead', 'manual'}:
            if not soft:
                self.release_remove_and_erase_qloud_host(host, target_project)
        return False

    def release_remove_and_erase_qloud_host(self, host: JyggalagHost, target_project: str):
        if target_project is None:
            raise Exception('Target project is none')
        if self.release_and_remove_qloud_host(host):
            if host.walle.project == target_project:
                if host.walle.state == 'free' and host.walle.status == 'ready':
                    return True
                return False
            try:
                self.logger.info('Erasing host [{}]{} in wall-e'.format(host.inv, host.name))
                self.walle.release_host(
                    host.walle,
                    target_project,
                    'ERASING FREE HOST'
                )
                return True
            except Exception as error:
                self.logger.exception('Error in erase operation')
                self.walle.force_status(host.walle, 'ready', 'ERASING FREE HOST')
                raise error
        return False

    def release_and_remove_qloud_host(self, host: JyggalagHost) -> bool:
        if self.qloud_host_released(host):
            if host.qloud is None:
                return True
            self.can_operate = False
            self.logger.info('{} Removing host from qloud'.format(host))
            self.qloud.remove_host(host.qloud)
        return False

    def qloud_host_released(self, host: JyggalagHost, target_state: str = 'DOWN') -> bool:
        if host.qloud is None:
            if not self.use_resolver:
                return True
            resolved_services = self.resolver.resolve_host(host.name).instances
            return len(resolved_services) == 0
        if (
                target_state is not None and target_state != '' and
                host.qloud.state != target_state and host.qloud.state not in qloud_down_states):
            self.logger.info('{} Host is not {} for release, changing state from {} to {}'.format(
                host,
                target_state,
                host.qloud.state,
                target_state))
            self.qloud.set_host_state(host.qloud, target_state, 'RELEASE {}'.format(str(datetime.now())))
            host.qloud.state = target_state
        if len(host.qloud.services) == 0:
            if not self.use_resolver:
                return True
            resolved_services = self.resolver.resolve_host(host.name).instances
            return len(resolved_services) == 0
        return False

    def switch_state(self, host: JyggalagHost, state: str):
        old_state = host.state
        host.state = state
        self.database.update_host_state(state, name=host.name, inv=host.inv)
        self.database.write_log(host.inv, host.name, old_state, state)
        self.logger.info(
            '{} SWITCH STATE STATE {} => {}'.format(host, old_state, host.state)
        )
