from infra.rtc.jyggalag.clients.qloud_api import QloudApi, datacenters
from infra.rtc.jyggalag.clients.walle_api import WalleApi
from infra.rtc.jyggalag.jyggalag_config import JyggalagConfig
from infra.rtc.jyggalag.jyggalag_data import JyggalagData
from infra.rtc.jyggalag.jyggalag_host import JyggalagHost
from infra.rtc.jyggalag.clients.service_resolver import ServiceResolver
from infra.rtc.jyggalag.jyggalag_machine import JyggalagMachine
from time import time, sleep


class JyggalagMind:

    def __init__(self, config: JyggalagConfig):
        self.config = config
        self.qloud = QloudApi(config)
        self.walle = WalleApi(config)
        self.resolver = ServiceResolver(config)
        if config.db is not None:
            self.db = JyggalagData(config)
        self.machine = JyggalagMachine(
            config=config,
            qloud=self.qloud,
            walle=self.walle,
            resolver=self.resolver,
            database=self.db
        )
        self.tick_time = int(config.redeploy.get('tick_time', 120))
        self.buffer_overcommit = int(config.redeploy.get('buffer_overcommit', 10))
        self.special_buffer_overcommit = config.redeploy.get('special_buffer_overcommit', dict())
        self.target_segments = config.redeploy.get('qloud_target_segments', None)
        self.state_timeout = int(config.redeploy.get('buffer_overcommit', 10))
        self.logger = config.logger
        self.target_dcs = datacenters
        self.readonly = config.readonly

        self.active = False
        self.lock_acquired = False
        self.wait_lock = False
        self.last_exec_time = 0
        self.last_hosts_errors = 0
        self.last_global_errors = 0
        self.lock_name = 'ALL'
        self.last_qloud_hosts = list()

        self.timeout_releasing_services = dict()
        self.releasing_services = dict()

    def shutdown(self, signal, frame):
        self.active = False
        exit(0)

    def get_hosts(self, hosts: list):
        host_set = set(hosts)
        walle_list = self.walle.get_hosts(hosts)
        id_walle_map = dict()
        name_walle_map = dict()
        unresolved = list()
        for walle_host in walle_list:
            id_walle_map[str(walle_host.inv)] = walle_host
            name_walle_map[walle_host.name] = walle_host
        for host in host_set:
            if host not in id_walle_map and host not in name_walle_map:
                unresolved.append(host)
        result_map = dict()
        for name in name_walle_map.keys():
            result_map[name] = JyggalagHost()
            result_map[name].walle = name_walle_map[name]

        qloud_request = list()
        for name in name_walle_map.keys():
            qloud_request.append(name)
        for unknown in unresolved:
            qloud_request.append(unknown)

        for qloud_host in self.qloud.find_hosts(qloud_request):
            name = qloud_host.name
            if name not in result_map:
                result_map[name] = JyggalagHost()
            result_map[name].qloud = qloud_host
        result = list()
        for name in result_map.keys():
            host = result_map[name]
            if host.walle is not None:
                host.inv = host.walle.inv
                host.name = host.walle.name
            if host.qloud is not None:
                host.name = host.qloud.name
            result.append(host)
        return result

    def get_segment_overcommit(self, segment: str) -> int:
        if self.special_buffer_overcommit is not None and segment in self.special_buffer_overcommit:
            return int(self.special_buffer_overcommit[segment])
        return self.buffer_overcommit

    def get_timeout_hosts(self):
        hosts = self.machine.get_hosts()
        result = list()
        for host in hosts:
            if int(time()) - host.update_time > self.state_timeout:
                result.append(host)
        return result

    def fill_releasing_services(self, hosts):
        services = dict()
        timeout_services = dict()
        for host in hosts:
            if host.qloud is None:
                continue
            if host.state in {'QLOUD_MOVE_RELEASING', 'QLOUD_REDEPLOY_RELEASING', 'QLOUD_REDEPLOY_BUFFER_RELEASING'}:
                timeout = int(time()) - host.update_time > self.state_timeout
                for slot in host.qloud.services.keys():
                    service_name = self.qloud.get_service_link(host.qloud.services[slot].get_full_env_name())
                    if service_name not in services:
                        services[service_name] = list()
                    services[service_name].append(host.name)
                    if timeout:
                        if service_name not in timeout_services:
                            timeout_services[service_name] = list()
                        timeout_services[service_name].append(host.name)
        self.timeout_releasing_services = timeout_services
        self.releasing_services = services

    def get_penging_hosts(self):
        hosts = self.machine.get_hosts()
        result = list()
        for host in hosts:
            if (
                    host.state == 'QLOUD_MOVE_RELEASE_PENDING' or
                    host.state == 'QLOUD_REDEPLOY_RELEASE_PENDING' or
                    host.state == 'QLOUD_REDEPLOY_BUFFER_RELEASE_PENDING'
            ):
                result.append(host)
        return result

    def get_qloud_redeploy_processing(self, hosts_data, segment: str, dc: str):
        result = list()
        for host in hosts_data:
            if host.walle is None:
                continue
            if (
                    host.segment == segment and
                    host.walle.dc.upper() == dc.upper() and(
                        host.state == 'QLOUD_REDEPLOY_ERASING' or
                        host.state == 'QLOUD_REDEPLOY_RELEASING' or
                        host.state == 'QLOUD_REDEPLOY_RELEASE_PENDING' or
                        host.state == 'QLOUD_REDEPLOY_PREPARING' or
                        host.state == 'QLOUD_REDEPLOY_ADDING'
                    )
            ):
                result.append(host)
        return result

    def get_qloud_redeploy_unused_buffer(self, redeploy_data, segment: str, dc: str):
        # active buffer in current SEGMENT/DC
        active_buffer = list()
        # processing or queued hosts for redeploy
        processing = list()
        result = list()
        for host in redeploy_data:
            if host.walle is None:
                continue
            if host.walle.dc.upper() == dc.upper():
                if host.state == 'QLOUD_REDEPLOY_BUFFER_BUSY' and host.qloud.segment == segment:
                    active_buffer.append(host)
                elif host.segment == segment and (
                        host.state == 'QLOUD_REDEPLOY_ERASING' or
                        host.state == 'QLOUD_REDEPLOY_RELEASING' or
                        host.state == 'QLOUD_REDEPLOY_RELEASE_PENDING' or
                        host.state == 'QLOUD_REDEPLOY_PREPARING' or
                        host.state == 'QLOUD_REDEPLOY_ADDING' or
                        host.state == 'QLOUD_REDEPLOY_QUEUED'
                ):
                    processing.append(host)
        delta = len(active_buffer) - len(processing) - self.get_segment_overcommit(segment) - 5
        if delta > 0:
            for i in range(delta):
                result.append(active_buffer[i])
        elif len(processing) == 0:
            result = active_buffer
        return result

    def get_qloud_redeploy_process_next(self, redeploy_data, segment: str, dc: str):
        processing = list()
        queued = list()
        active_buffer = list()
        result = list()

        for host in redeploy_data:
            if host.walle is None:
                continue
            if host.walle.dc.upper() == dc.upper():
                if host.segment == segment and host.state == 'QLOUD_REDEPLOY_QUEUED':
                    queued.append(host)
                if host.state == 'QLOUD_REDEPLOY_BUFFER_BUSY' and host.qloud.segment == segment:
                    active_buffer.append(host)
                elif host.segment == segment and (
                        host.state == 'QLOUD_REDEPLOY_ADDING' or
                        host.state == 'QLOUD_REDEPLOY_PREPARING' or
                        host.state == 'QLOUD_REDEPLOY_RELEASING' or
                        host.state == 'QLOUD_REDEPLOY_RELEASE_PENDING' or
                        host.state == 'QLOUD_REDEPLOY_ERASING'
                ):
                    processing.append(host)
        delta = len(active_buffer) - len(processing) - self.get_segment_overcommit(segment)
        if len(queued) <= delta:
            return queued
        elif delta > 0:
            for i in range(delta):
                result.append(queued[i])
        return result

    def get_qloud_redeploy_buffer_apply_next(self, redeploy_data, segment: str, dc: str):
        # active buffer in current SEGMENT/DC
        active_buffer = list()

        free_buffer = list()
        # processing or queued hosts for redeploy
        processing = list()
        result = list()
        for host in redeploy_data:
            if host.walle is None:
                continue
            if (host.walle.dc.upper() == dc.upper() and
                    host.qloud is not None):
                if ((host.state in {'QLOUD_REDEPLOY_BUFFER_BUSY', 'QLOUD_REDEPLOY_BUFFER_PENDING_BUSY'}) and
                        host.qloud.segment == segment):
                    active_buffer.append(host)
                elif host.state == 'QLOUD_REDEPLOY_BUFFER_FREE':
                    free_buffer.append(host)
                elif host.segment == segment and (
                        host.state == 'QLOUD_REDEPLOY_ERASING' or
                        host.state == 'QLOUD_REDEPLOY_RELEASE_PENDING' or
                        host.state == 'QLOUD_REDEPLOY_RELEASING' or
                        host.state == 'QLOUD_REDEPLOY_PREPARING' or
                        host.state == 'QLOUD_REDEPLOY_ADDING' or
                        host.state == 'QLOUD_REDEPLOY_QUEUED'
                ):
                    processing.append(host)
        if len(processing) == 0:
            return result
        delta = len(processing) - len(active_buffer) + self.get_segment_overcommit(segment)
        if delta > 0:
            if len(free_buffer) > delta:
                for i in range(delta):
                    result.append(free_buffer[i])
            else:
                result = free_buffer
        return result

    def qloud_shedule(self):
        buffer_to_free = list()
        hosts_to_redeploy = list()

        hosts_data = self.machine.get_hosts()
        self.fill_releasing_services(hosts_data)
        if self.readonly:
            return
        for segment in self.target_segments:
            for dc in self.target_dcs:
                buffer_to_apply = self.get_qloud_redeploy_buffer_apply_next(hosts_data, segment, dc)
                for host in buffer_to_apply:
                    try:
                        self.machine.database.update_segment(host, segment)
                        self.machine.switch_state(host, 'QLOUD_REDEPLOY_BUFFER_PENDING_BUSY')
                    except Exception:
                        self.logger.exception("Error during Apply buffer schedule")
                        self.last_hosts_errors += 1

                buffer_to_free.extend(self.get_qloud_redeploy_unused_buffer(hosts_data, segment, dc))
                hosts_to_redeploy.extend(self.get_qloud_redeploy_process_next(hosts_data, segment, dc))
        for host in buffer_to_free:
            target_segment = '{}.{}'.format(host.qloud.installation, self.machine.buffer_segment_name)
            self.machine.database.update_segment(host, target_segment)
            self.machine.switch_state(host, 'QLOUD_REDEPLOY_BUFFER_RELEASE_PENDING')
        for host in hosts_to_redeploy:
            self.machine.switch_state(host, 'QLOUD_REDEPLOY_RELEASE_PENDING')

    def rotate_forever(self):
        self.active = True
        while self.active:
            try:
                self.rotate()
                for i in range(self.tick_time):
                    if not self.active:
                        break
                    sleep(1)
            except Exception:
                self.last_global_errors += 1
                pass

    def wait_for_lock(self, resource: str = '*'):
        if resource is None:
            return
        while not self.db.get_lock(resource) and self.active:
            sleep(1)

    def rotate(self):
        self.last_global_errors = 0
        self.last_hosts_errors = 0
        self.wait_lock = True
        self.wait_for_lock(self.lock_name)
        if not self.active:
            self.db.release_lock(self.lock_name)
            return
        self.wait_lock = False
        start = int(time())
        self.lock_acquired = True
        try:
            self.qloud_shedule()
        except Exception:
            self.logger.exception("Error during schedule")
            self.last_global_errors += 1
        try:
            self.machine.tick()
        except Exception:
            self.logger.exception("Error during state machine tick")
            self.last_global_errors += 1
        self.last_global_errors += self.machine.last_global_errors
        self.last_hosts_errors += self.machine.last_host_errors
        self.last_exec_time = int(time()) - start
        self.db.release_lock(self.lock_name)
        self.lock_acquired = False
