package main

import (
	"errors"
	"flag"
	"math/rand"
	"net"
	"net/http"
	"sync"
	"sync/atomic"
	"time"

	"github.com/grpc-ecosystem/grpc-gateway/runtime"
	"golang.org/x/net/context"
	"google.golang.org/grpc"
	"google.golang.org/grpc/keepalive"

	"a.yandex-team.ru/infra/rtc/loadgen/pkg"
	"a.yandex-team.ru/library/go/core/log"
)

var (
	grpcPort        = flag.String("grpc-endpoint", ":8787", "endpoint of GRPC API")
	grpcConnectPort = flag.String("grpc-connect-endpoint", "localhost:8787", "endpoint of GRPC API")
	restPort        = flag.String("rest-endpoint", ":8080", "endpoint of http REST API")
	configPath      = flag.String("config-path", "config.yaml", "path to config")
)

func startClients(clientPool *pkg.ClientPool, config *pkg.LoadgenConfig) error {
	var fails uint64
	var waiter sync.WaitGroup
	r := rand.New(rand.NewSource(time.Now().Unix()))
	for idx, endpointIdx := range r.Perm(len(config.Endpoints)) {
		if config.MaxClients > 0 && idx >= config.MaxClients {
			break
		}
		endpoint := config.Endpoints[endpointIdx]
		for _, payloadSize := range config.PayloadSizes {
			waiter.Add(1)
			go func(endpoint string, payloadSize int) {
				defer waiter.Done()
				if err := clientPool.ClientLoop(endpoint, payloadSize); err != nil {
					atomic.AddUint64(&fails, 1)
					pkg.Logger.Error("can't start client", log.Error(err))
					clientPool.Stop()
				}
			}(endpoint, payloadSize)
		}
	}
	waiter.Wait()
	if atomic.LoadUint64(&fails) > 0 {
		return errors.New("can't start some clients")
	} else {
		return nil
	}
}

func main() {
	flag.Parse()

	ctx := context.Background()
	ctx, cancel := context.WithCancel(ctx)
	defer cancel()

	config, err := pkg.GetConfig(*configPath)
	if err != nil {
		pkg.Logger.Error("failed to read config", log.Error(err))
		return
	}

	clientPool := pkg.NewClientPool(ctx, config)

	lis, err := net.Listen("tcp6", *grpcPort)
	if err != nil {
		pkg.Logger.Error("failed to listen", log.Error(err))
		return
	}

	mux := runtime.NewServeMux()
	opts := []grpc.DialOption{
		grpc.WithInsecure(),
		grpc.WithReadBufferSize(256 * 1024),
		grpc.WithWriteBufferSize(256 * 1024),
	}
	if err := pkg.RegisterLoadgenHandlerFromEndpoint(ctx, mux, *grpcConnectPort, opts); err != nil {
		pkg.Logger.Error("failed to register http endpoint", log.Error(err))
		return
	}
	pkg.NewUnistatHandle("stats", mux)

	server := grpc.NewServer(
		grpc.UnaryInterceptor(grpc.UnaryServerInterceptor(pkg.RequestInterceptor)),
		grpc.KeepaliveParams(keepalive.ServerParameters{
			MaxConnectionAge: time.Duration(config.ConnectionLifetime) * time.Millisecond,
		}),
	)
	loadgen := pkg.NewLoadgenImpl(config)
	loadgen.Register(server)

	go func() {
		if err := http.ListenAndServe(*restPort, mux); err != nil {
			pkg.Logger.Error("failed to serve http proxy", log.Error(err))
			server.Stop()
		}
	}()

	go func() {
		if err := startClients(clientPool, config); err != nil {
			pkg.Logger.Error("failed to start clients", log.Error(err))
			server.Stop()
		}
	}()

	if err := server.Serve(lis); err != nil {
		pkg.Logger.Error("failed to serve", log.Error(err))
		return
	}
}
