package pkg

import (
	"crypto/rand"
	random "math/rand"
	"sync/atomic"
	"time"

	"golang.org/x/net/context"
	"google.golang.org/grpc"

	pb "a.yandex-team.ru/infra/rtc/loadgen/api"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/unistat"
	"a.yandex-team.ru/library/go/yandex/unistat/aggr"
)

type ClientPool struct {
	latencyHist  *LatencyHistogram
	errorCounter *unistat.Numeric
	ctx          context.Context
	closed       int32
}

func NewClientPool(ctx context.Context, config *LoadgenConfig) (client *ClientPool) {
	client = &ClientPool{
		latencyHist:  NewLatencyHistogram("client", config),
		errorCounter: unistat.NewNumeric("client_errors", 10, aggr.Counter(), unistat.Sum),
		ctx:          ctx,
		closed:       0,
	}
	return
}

func (p *ClientPool) ClientLoop(address string, payloadSize int) error {
	//nolint:SA1019
	conn, err := grpc.Dial(
		address,
		grpc.WithInsecure(),
		grpc.WithBackoffMaxDelay(time.Second),
		grpc.WithReadBufferSize(256*1024),
		grpc.WithWriteBufferSize(256*1024),
	)
	if err != nil {
		return err
	}
	defer conn.Close()

	payload := make([]byte, payloadSize)
	_, err = rand.Read(payload)
	if err != nil {
		return err
	}

	client := pb.NewLoadgenClient(conn)
	payloadRequest := &pb.PayloadRequest{
		Payload: payload,
	}

	ping := func() {
		ctx, cancel := context.WithTimeout(p.ctx, time.Second)
		defer cancel()

		payloadRequest.Timestamp = uint64(time.Now().UnixNano())
		response, err := client.Payload(ctx, payloadRequest)
		if err != nil {
			Logger.Error("Can't request server", log.String("address", address), log.Error(err))
			p.errorCounter.Update(1)
			return
		}

		timestampDelta := AbsoluteDelta(time.Now().UnixNano(), int64(response.Timestamp))
		p.latencyHist.Update(len(response.Payload), float64(timestampDelta)/1000.0)
	}

	for {
		closed := atomic.LoadInt32(&p.closed)
		if closed != 0 {
			return nil
		} else {
			ping()
			time.Sleep(5*time.Millisecond + time.Duration(random.Int63n(int64(20*time.Millisecond))))
		}
	}
}

func (p *ClientPool) Stop() {
	atomic.StoreInt32(&p.closed, 1)
}
