package pkg

import (
	"fmt"

	"a.yandex-team.ru/library/go/yandex/unistat"
	"a.yandex-team.ru/library/go/yandex/unistat/aggr"
)

var (
	US          = 1.0
	MS          = 1000.0 * US
	S           = 1000.0 * MS
	timeBuckets = []float64{
		0,
		100 * US,
		200 * US,
		300 * US,
		400 * US,
		500 * US,
		600 * US,
		700 * US,
		800 * US,
		900 * US,
		1 * MS,
		2 * MS,
		3 * MS,
		4 * MS,
		5 * MS,
		6 * MS,
		7 * MS,
		8 * MS,
		9 * MS,
		10 * MS,
		11 * MS,
		12 * MS,
		13 * MS,
		14 * MS,
		15 * MS,
		16 * MS,
		17 * MS,
		18 * MS,
		19 * MS,
		20 * MS,
		30 * MS,
		40 * MS,
		50 * MS,
		60 * MS,
		70 * MS,
		80 * MS,
		90 * MS,
		100 * MS,
		500 * MS,
		1 * S,
	}
)

type LatencyBucket struct {
	upperBound int
	requests   *unistat.Numeric
	histogram  *unistat.Histogram
}

type LatencyHistogram struct {
	buckets         []*LatencyBucket
	overallRequests *unistat.Numeric
	overallLatency  *unistat.Histogram
}

func NewLatencyHistogram(prefix string, config *LoadgenConfig) (h *LatencyHistogram) {
	h = &LatencyHistogram{
		buckets:         make([]*LatencyBucket, 0),
		overallRequests: unistat.NewNumeric(fmt.Sprintf("%s_requests", prefix), 10, aggr.Counter(), unistat.Sum),
		overallLatency:  unistat.NewHistogram(fmt.Sprintf("%s_latency", prefix), 10, aggr.Histogram(), timeBuckets),
	}
	unistat.Register(h.overallRequests)
	unistat.Register(h.overallLatency)
	for _, upperBound := range config.PayloadSizes {
		requests := unistat.NewNumeric(fmt.Sprintf("%s_requests_payload%d", prefix, upperBound), 10, aggr.Counter(), unistat.Sum)
		histogram := unistat.NewHistogram(fmt.Sprintf("%s_latency_payload%d", prefix, upperBound), 10, aggr.Histogram(), timeBuckets)
		unistat.Register(requests)
		unistat.Register(histogram)
		h.buckets = append(h.buckets, &LatencyBucket{
			upperBound: upperBound,
			requests:   requests,
			histogram:  histogram,
		})
	}
	return
}

func (h *LatencyHistogram) Update(payloadSize int, value float64) {
	h.overallRequests.Update(1)
	h.overallLatency.Update(value)
	for _, bucket := range h.buckets {
		if bucket.upperBound >= payloadSize {
			bucket.requests.Update(1)
			bucket.histogram.Update(value)
			break
		}
	}
}
