# coding: utf-8

from __future__ import absolute_import, print_function

import os
import sys
import logging
from collections import defaultdict
from datetime import datetime, timedelta

import click
from cached_property import cached_property
from walle_api import WalleClient
from startrek_client import Startrek
from dateutil.parser import parse as dt_parse
from dateutil.tz import tzutc
from yaml import safe_dump

MAINTENANCE_TAG = "rtc_maintenance_expired"


class WalleHost(object):

    __slots__ = ["fqdn", "project", "status", "ticket"]

    def __init__(self):
        self.fqdn = None
        self.project = None
        self.status = None
        self.ticket = None


class CliContext(object):

    def __init__(self):
        self.dry_run = True
        self.walle_token = None
        self.startrek_token = None

    def _read_token(self, path):
        try:
            with open(path) as stream:
                return stream.read().strip()
        except Exception:
            logging.exception("Can't get token from %s", path)
        return None

    def _read_st_token(self):
        return self._read_token(os.path.expanduser("~/.st/token"))

    def _read_walle_token(self):
        return self._read_token(os.path.expanduser("~/.wall-e/access_token"))

    @cached_property
    def walle_client(self):
        return WalleClient(access_token=self.walle_token or self._read_walle_token())

    @cached_property
    def st_client(self):
        return Startrek(useragent="rtc walle maintenance", token=self.startrek_token or self._read_st_token())


def build_ticket_map(walle_client, st_client):
    ticket_map = {}
    host_map = defaultdict(WalleHost)
    for host in walle_client.iter_hosts(fields=["name", "project", "status", "state_expire"], tags=["rtc"], state=["maintenance"]):
        walle_host = host_map[host["name"]]
        walle_host.fqdn = host["name"]
        walle_host.project = host["project"]
        walle_host.status = host["status"]

        state_expire = host.get("state_expire") or {}
        ticket = state_expire.get("ticket")
        if ticket:
            if ticket not in ticket_map:
                ticket_map[ticket] = st_client.issues[ticket]
            walle_host.ticket = ticket_map[ticket]

    comment_map = {}
    hosts_without_ticket = []
    hosts_in_ticket = defaultdict(list)
    for host in host_map.values():
        if not host.ticket or not hasattr(host.ticket, "deadline"):
            hosts_without_ticket.append(host)
            continue
        deadline = None
        if host.ticket.deadline:
            deadline = dt_parse(host.ticket.deadline)
        if deadline and datetime.now() <= deadline:
            continue
        if host.ticket.key not in comment_map:
            comment_map[host.ticket.key] = list(host.ticket.comments)
        comments = comment_map[host.ticket.key]
        if comments:
            last_activitity = max([dt_parse(comment.updatedAt) for comment in comments])
        else:
            last_activitity = dt_parse(host.ticket.updatedAt)
        ttl = 7
        if host.ticket.queue.key == "RUNTIMECLOUD" and host.ticket.status.key == "blocked":
            ttl = 30
        elif host.ticket.queue.key != "RUNTIMECLOUD":
            ttl = 14
        if (datetime.now(tz=tzutc()) - last_activitity) <= timedelta(days=ttl):
            continue
        hosts_in_ticket[host.ticket.key].append(host)

    return hosts_in_ticket, hosts_without_ticket


@click.group()
@click.option('--walle-token', envvar='WALLE_OAUTH', default='',
              metavar='OAUTH', help='OAuth token for Wall-E.')
@click.option('--startrek-token', envvar='STARTREK_OAUTH', default='',
              metavar='OAUTH', help='OAuth token for Startrek.')
@click.option('--dry-run/--no-dry-run', default=True)
@click.pass_context
def cli(ctx, walle_token, startrek_token, dry_run):
    ctx.obj.dry_run = dry_run
    ctx.obj.walle_token = walle_token
    ctx.obj.startrek_token = startrek_token
    logging.info("dry run: %r", ctx.obj.dry_run)


@cli.command()
@click.pass_context
def maintenance(ctx):
    hosts_in_ticket, hosts_without_ticket = build_ticket_map(ctx.obj.walle_client, ctx.obj.st_client)

    result = []
    tickets = []
    for hosts in sorted(hosts_in_ticket.values(), reverse=True, key=lambda x: len(x)):
        hosts.sort(key=lambda x: x.fqdn)
        ticket = hosts[0].ticket
        result.append({
            "ticket_link": "http://st.yandex-team.ru/{}".format(ticket.key),
            "host_count": len(hosts),
            "host_links": ["https://wall-e.yandex-team.ru/host/{}".format(host.fqdn) for host in hosts]
        })
        tickets.append(ticket)

    safe_dump({
        "tickets": result,
        "without_tickets": len(hosts_without_ticket)
    }, sys.stdout, default_flow_style=False)

    for ticket in tickets:
        current_tags = set(ticket.tags)
        new_tags = set(current_tags)
        new_tags.add(MAINTENANCE_TAG)
        if current_tags != new_tags:
            if not ctx.obj.dry_run:
                ticket.update(tags=sorted(new_tags))
            else:
                print("{} should be updated (add): {}".format(ticket.key, sorted(new_tags)))

    for ticket in ctx.obj.st_client.issues.find(filter={'tags': [MAINTENANCE_TAG], 'resolution': 'empty()'}, per_page=100):
        if ticket.key not in hosts_in_ticket:
            current_tags = set(ticket.tags)
            new_tags = set(current_tags)
            new_tags.remove(MAINTENANCE_TAG)
            if current_tags != new_tags:
                if not ctx.obj.dry_run:
                    ticket.update(tags=sorted(new_tags))
                else:
                    print("{} should be updated (remove): {}".format(ticket.key, sorted(new_tags)))


@cli.command()
@click.argument('ticket', nargs=1, required=True)
@click.pass_context
def hosts(ctx, ticket):
    hosts_in_ticket, _ = build_ticket_map(ctx.obj.walle_client, ctx.obj.st_client)
    hosts = hosts_in_ticket.get(ticket, [])
    safe_dump({
        "ticket_link": "http://st.yandex-team.ru/{}".format(ticket),
        "host_count": len(hosts),
        "host_links": ["https://wall-e.yandex-team.ru/host/{}".format(host.fqdn) for host in hosts]
    }, sys.stdout, default_flow_style=False)
