import argparse
import errno
import logging
import socket
import sys

import os

from infra.rtc.nodeinfo.lib import node_info
from infra.rtc.nodeinfo.lib import statefile
from infra.rtc.nodeinfo.lib.reporters import yp
from infra.ya_salt.lib import walle, pbutil
from infra.ya_salt.proto import ya_salt_pb2

log = logging.getLogger('nodeinfo')


def parse_args(argv):
    parser = argparse.ArgumentParser()
    parser.add_argument('-v', '--version', action='store_true', default=False)
    parser.add_argument('-l', '--log-level', default='INFO', help='Set logging level (default=INFO)')
    parser.add_argument('--dry-run', action='store_true', default=False,
                        help='Collect HostmanStatus and print to stdout ./as json without reporting to yp')
    parser.add_argument('--cpu-model-override', default=None,
                        help='CPU model override string (see https://st.yandex-team.ru/HOSTMAN-1137 for details')
    return parser.parse_args(argv)


def main(argv):
    if argv.version is True:
        from library.python import svn_version
        print('revision: {}'.format(svn_version.svn_revision()))
        return None
    hostname = socket.gethostname()
    grains, err = walle.grains(hostname)
    if err is not None:
        return err
    status = ya_salt_pb2.HostmanStatus()
    node_info.run_node_info(grains['walle_tags'], status.node_info)

    # apply cpu model override
    if argv.cpu_model_override is not None:
        status.node_info.cpu_info.model_name = argv.cpu_model_override

    if status.node_info.ok.status != 'True':
        print(pbutil.pb_to_json(status))
        return "Error gathering node_info: {}".format(status.node_info.ok.message)
    if argv.dry_run is True:
        print(pbutil.pb_to_json(status))
        return None
    err = yp.send_nodeinfo(hostname, grains['walle_tags'], grains['walle_dc'], status)
    if err:
        return "Cannot send node_info to yp: {}".format(err)
    return None


def run_main(args):
    try:
        return main(args)
    except EnvironmentError as e:
        if e.errno != errno.EPIPE:
            log.exception('Unexpected error.')
            return 'failed to run main with unexpected error: {}'.format(e)
        else:
            return None
    except KeyboardInterrupt:
        log.info('=== Interrupted, quitting... ===')
        return None
    except Exception as e:
        log.exception('Unexpected error.')
        return 'failed to run main with unexpected error: {}'.format(e)


if __name__ == "__main__":
    args = parse_args(sys.argv[1:])
    del logging.getLogger().handlers[:]
    # Do not print timestamps as we log to stdout, which is collected by journald.
    logging.basicConfig(level=getattr(logging, args.log_level, logging.INFO),
                        format='%(name)s - %(levelname)s - %(message)s')
    err = run_main(args)
    try:
        sys.stdout.flush()
        sys.stderr.flush()
    except EnvironmentError as e:
        if e.errno == errno.EPIPE:
            pass
        raise
    ret = 0 if err is None else 1
    if not args.dry_run:
        ret = statefile.retcode_from_run_error(err)
    # Do not cleanup interpreter state: we want to exit fast.
    os._exit(ret)
