import argparse
import errno
import logging
import sys
import os
from infra.rtc.nodeinfo.lib.sox import report
from infra.rtc.nodeinfo.lib.sox import server

log = logging.getLogger('sox-report')


def parse_args(argv):
    parser = argparse.ArgumentParser()
    parser.add_argument('-v', '--version', action='store_true', default=False)
    parser.add_argument('-l', '--log-level', default='INFO', help='Set logging level (default=INFO)')
    parser.add_argument('-t', '--token', help='Yp token (by default trying to get from YP_TOKEN env var)')
    parser.add_argument('-a', '--all-hosts', required=False, default=False, action='store_true',
                              help='Get info for all hosts in group')
    parser.add_argument('-o', '--output-format', help="Output format: console or html", default='console')
    parser.add_argument('-d', '--output-directory', help="Output directory for html formatter, default = cwd")

    sel_group = parser.add_mutually_exclusive_group()
    node_group = sel_group.add_argument_group()
    node_group.add_argument('-c', '--cluster', help='Yp cluster to query (possible clusters: sas,sas-test,man-pre,vla,man,iva,myt,xdc), used with --node')
    node_group.add_argument('-n', '--node', help='Node fqdn (used with --cluster)')
    sel_group.add_argument('-G', '--gencfg', help='Get SOX info for GENCFG group')
    sel_group.add_argument('-W', '--walle-project', help='Get SOX info for Wall-e project')
    sel_group.add_argument('-f', '--nanny-service', help='Get SOX info for Nanny service')

    parser.add_argument('--server', help='Run sox_report server on port', default=None, type=int)

    return parser.parse_args(argv)


def main(argv):
    if argv.version is True:
        from library.python import svn_version
        print('revision: {}'.format(svn_version.svn_revision()))
        return True
    results = []
    err = None
    token = report.yp_token(argv)
    if argv.server:
        exit(server.run(token, argv.server))
    if argv.node:
        results, err = report.lookup_node(token, argv.cluster, argv.node)
    elif argv.gencfg:
        results, err = report.lookup_gencfg(token, argv.gencfg)
    elif argv.walle_project:
        results, err = report.lookup_walle(token, argv.walle_project)
    elif argv.nanny_service:
        results, err = report.lookup_nanny(token, argv.nanny_service)
    else:
        log.fatal("Dont know what to do")
        exit(1)
    if err is not None:
        log.error('Failed to do lookup: {}'.format(err))
        exit(1)
    if argv.output_format=='console':
        log.info("setting formatter to console")
        formatter = report.format_console
    elif argv.output_format=='html':
        log.info("setting formatter to html")
        formatter = report.format_html_to_dir
    else:
        log.info("using fallback formatter console")
        formatter = report.format_console
    formatter(argv.output_directory if argv.output_directory else os.getcwd(), results)


if __name__ == "__main__":
    args = parse_args(sys.argv[1:])
    del logging.getLogger().handlers[:]
    logging.basicConfig(level=getattr(logging, args.log_level, logging.INFO),
                        format='%(asctime)s - %(name)s - %(levelname)s - %(message)s')
    try:
        ok = main(args)
    except EnvironmentError as e:
        if e.errno != errno.EPIPE:
            log.exception('Unexpected error.')
            ok = False
        else:
            ok = True
    except KeyboardInterrupt as e:
        log.info('=== Interrupted, quitting... ===')
        ok = True
    except SystemExit as e:
        ok = False
        os._exit(e.code)
    except Exception as e:
        log.exception('Unexpected error.')
        ok = False
    try:
        sys.stdout.flush()
        sys.stderr.flush()
    except EnvironmentError as e:
        if e.errno == errno.EPIPE:
            pass
        raise
    os._exit(0 if ok else 1)
    raise SystemExit(0 if ok else 1)
