import argparse
import errno
import logging
import sys
import os
import yp.client

from infra.ya_salt.proto import ya_salt_pb2
from infra.ya_salt.lib import pbutil

log = logging.getLogger('yp-nodeinfo')


def parse_args(argv):
    parser = argparse.ArgumentParser()
    parser.add_argument('-v', '--version', action='store_true', default=False)
    parser.add_argument('-l', '--log-level', default='INFO', help='Set logging level (default=INFO)')
    parser.add_argument('-c', '--cluster', required=True, help='Yp cluster to query (possible clusters: sas,sas-test,man_pre,vla,man,iva,myt,xdc)')
    parser.add_argument('-n', '--node', required=True, help='Node fqdn')
    parser.add_argument('-t', '--token', help='Yp token (by default trying to get from YP_TOKEN env var)')
    return parser.parse_args(argv)


def main(argv):
    if argv.version is True:
        from library.python import svn_version
        print('revision: {}'.format(svn_version.svn_revision()))
        return True
    if argv.token:
        token = argv.token
    else:
        token = os.getenv('YP_TOKEN')
    client = yp.client.YpClient(argv.cluster, config={"token": token})
    node_info = client.select_objects("node", filter='[/meta/id]="{}"'.format(argv.node), selectors=['/status/host_manager'])[0][0]
    status = ya_salt_pb2.HostmanStatus()
    status.MergeFromString(node_info['value'])
    print(pbutil.pb_to_json(status))


if __name__ == "__main__":
    args = parse_args(sys.argv[1:])
    del logging.getLogger().handlers[:]
    logging.basicConfig(level=getattr(logging, args.log_level, logging.INFO),
                        format='%(asctime)s - %(name)s - %(levelname)s - %(message)s')
    try:
        ok = main(args)
    except EnvironmentError as e:
        if e.errno != errno.EPIPE:
            log.exception('Unexpected error.')
            ok = False
        else:
            ok = True
    except KeyboardInterrupt as e:
        log.info('=== Interrupted, quitting... ===')
        ok = True
    except SystemExit as e:
        ok = False
        os._exit(e.code)
    except Exception as e:
        log.exception('Unexpected error.')
        ok = False
    try:
        sys.stdout.flush()
        sys.stderr.flush()
    except EnvironmentError as e:
        if e.errno == errno.EPIPE:
            pass
        raise
    os._exit(0 if ok else 1)
    raise SystemExit(0 if ok else 1)
