import io
import os

from infra.ya_salt.lib import fileutil


class CPUInfo(object):
    def __init__(self):
        self.cpu = -1
        self.cores = -1
        self.core_id = ''
        self.vendor_id = ''
        self.family = ''
        self.model = ''
        self.stepping = ''
        self.model_name = ''
        self.mhz = 0.0
        self.cache_size = 0
        self.flags = []
        self.microcode = ''
        self.physical_id = ''


def sys_cpu_path(cpu, relpath, syspath='/sys'):
    return os.path.join(syspath, 'devices/system/cpu/cpu{}'.format(cpu), relpath)


def finish_cpu_info(c):
    if not c.core_id:
        buf, err = fileutil.read_file(sys_cpu_path(c.cpu, "topology/core_id"))
        if err is None:
            c.core_id = buf.strip()
    # Override the value of c.mhz with cpufreq/cpuinfo_max_freq regardless
    # of the value from /proc/cpuinfo because latter is current frequency
    # which depends on usage/governor/etc.
    buf, err = fileutil.read_file(sys_cpu_path(c.cpu, "cpufreq/cpuinfo_max_freq"))
    if err is not None:
        return None
    try:
        v = float(buf.rstrip())
    except Exception as e:
        return str(e)
    c.mhz = v / 1000.0  # v is in kHz
    if c.mhz > 9999.0:
        c.mhz //= 1000.0  # v is in Hz


def cpu_info_from_buf(buf):
    f = io.BytesIO(buf)
    ret = []
    c = CPUInfo()
    p_name = ''
    for l in f:
        fields = l.split(':')
        if len(fields) < 2:
            continue
        k = fields[0].strip()
        v = fields[1].strip()
        if k == 'Processor':
            p_name = v
        elif k == 'processor':
            if c.cpu >= 0:
                finish_cpu_info(c)
                ret.append(c)
            c = CPUInfo()
            c.cores = 1
            c.model_name = p_name
            try:
                t = int(v, 10)
            except Exception as e:
                return None, str(e)
            c.cpu = t
        elif k in ['vendorId', 'vendor_id']:
            c.vendor_id = v
        elif k == 'cpu family':
            c.family = v
        elif k == 'model':
            c.model = v
        elif k in ['model name', 'cpu']:
            c.model_name = v
        elif k in ['stepping', 'revision']:
            val = v
            if k == 'revision':
                val = v.split('.')[0]
            try:
                t = int(val)
            except Exception as e:
                return None, str(e)
            c.stepping = t
        elif k in ['cpu MHz', 'clock']:
            # Don't treat as error, because we grab it from other source
            try:
                c.mhz = float(v.replace('MHz', '', 1))
            except Exception:
                continue
        elif k == 'cache size':
            try:
                t = int(v.replace(' KB', '', 1))
            except Exception as e:
                return None, str(e)
            c.cache_size = t
        elif k == 'physical id':
            c.physical_id = v
        elif k == 'core id':
            c.core_id = v
        elif k in ['flags', 'Features']:
            c.flags = v.split(' ')
            c.flags.sort()
        elif k == 'microcode':
            c.microcode = v
    if c.cpu >= 0:
        finish_cpu_info(c)
        ret.append(c)
    return ret, None


def cpu_info(path='/proc/cpuinfo'):
    buf, err = fileutil.read_file(path, max_size=1 * 1024 * 1024)
    if err is not None:
        return None, err
    # This part is has no tests :(
    # So to avoid breaking hostman - wrap in try/except for now
    try:
        return cpu_info_from_buf(buf)
    except Exception as e:
        return None, 'cpu_info_from_buf() failed: {}'.format(e)
